#!/usr/bin/env python
# -*- coding: utf-8 -*-
# py-indent-offset:4 -*-

#-------------------------------------------------------------------------------
# Name:             textctrl.py
# Purpose:          Custom text control
# Author:           Ecco
# Created:          2023
# Copyright:        ...
# License:          wxWindows license
# Version:          1.0.0
# Tags:             phoenix-port, py3-port
# ....
# Tested:           - Windows 10/11 | Python 3.11.9 |
# ....              wxPython 4.2.3 | wxWidgets 3.2.6
# ....
# ....              - Linux Mint 21 | Python 3.10.12 |
# ....              wxPython 4.2.1 gtk3 | wxWidgets 3.2.2.1
# ....
# ....              - MacOS Sequoia 15 | Python 3.12.4 |
# ....              wxPython 4.2.2 | wxWidgets 3.2.6
# ....
# Thanks to:        Cody Precord
# (A-Z)             ...
#-------------------------------------------------------------------------------

"""

1.0.0   First release

"""

#-------------------------------------------------------------------------------
# Import wxPython packages
#-------------------------------------------------------------------------------
import wx

#-------------------------------------------------------------------------------

# class TextCtrlPanel

#-------------------------------------------------------------------------------
        
class TextCtrlPanel(wx.Panel):
    def __init__(self, parent, id=wx.ID_ANY,
                 pos=wx.DefaultPosition, size=(200, 30),
                 value="",
                 brushColour=(wx.Colour(132, 178, 43, 0)),
                 penColour=wx.Colour(0, 0, 0, 255),
                 penThickness=1,
                 shadowColour=wx.Colour(255, 255, 255, 255),
                 rounded=4,
                 textColour=wx.Colour(0, 0, 0, 255),
                 style=wx.BORDER_NONE|wx.TRANSPARENT_WINDOW, name="name"):
        super(TextCtrlPanel, self).__init__(parent, id, pos, size=size,   
                                             style=style, name=name)

        # Attributes
        self._style = style
        self.value = value
        self.brushColour = brushColour     
        self.penColour = penColour
        self.penThickness = penThickness
        self.shadowColour = shadowColour
        self.rounded = rounded
        self.textColour = textColour
        self.is_text_enabled = True  # Text status

        # System font size
        font_sizes = self.GetFontSizes()

        selected_font_size = font_sizes[2]  # Select the third size from the list
        font = wx.SystemSettings.GetFont(wx.SYS_DEFAULT_GUI_FONT)
        font.SetPointSize(selected_font_size)  # Apply the selected size
        
        self.w, self.h = self.GetSize()
        
        # Create a default TextCtrl 
        self.text_ctrl = wx.TextCtrl(self, value="",
                                     size=(-1, 30),  
                                     style=wx.TE_LEFT |wx.TE_RICH2 |
                                           wx.TE_PROCESS_ENTER | wx.BORDER_NONE |
                                           wx.TRANSPARENT_WINDOW)
        self.text_ctrl.SetInsertionPoint(0)
        self.text_ctrl.SetMaxLength(32)
        self.text_ctrl.SetFont(font)
        message  = "This is custom TextCtrl"
        self.text_ctrl.SetToolTip(message)

        self.text_ctrl.SetForegroundColour(self.textColour)  
  
        if wx.Platform == "__WXGTK__":  # Linux Background Transparency
            self.text_ctrl.SetBackgroundColour(wx.Colour(242,242,242,0)) 
        elif wx.Platform == "__WXMAC__":  # MacOS Background Transparency
            self.text_ctrl.SetBackgroundColour(wx.Colour(242,242,242,0)) 
            
        self.text_ctrl.Bind(wx.EVT_CHECKBOX, self.GetValue)

        # Create a horizontal sizer to put the TextCtrl and the icon together 
        self.sizerTest = wx.BoxSizer(wx.HORIZONTAL)
        if wx.Platform == "__WXGTK__":
            self.sizerTest.Add(self.text_ctrl, 1, wx.EXPAND | wx.LEFT, 9)  
        elif wx.Platform == "__WXMSW__":
            self.sizerTest.Add(self.text_ctrl, 1, wx.EXPAND | wx.LEFT, 0)  
        elif wx.Platform == "__WXMAC__":
            self.sizerTest.Add(self.text_ctrl, 1, wx.EXPAND | wx.LEFT, 9)
        
        # Create a layout with a vertical sizer
        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(self.sizerTest, 0, wx.ALL | wx.EXPAND, 4)  # Offsets the text by 3 px

        self.SetSizer(sizer)

        self.Bind(wx.EVT_PAINT, lambda evt: self.Draw())
        self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground) # Background transparency 
        self.text_ctrl.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)  # Background transparency      

    def Draw(self):
        if self._style:
            dc = wx.PaintDC(self)
        else:
            dc = wx.AutoBufferedPaintDCFactory(self)

        gc = wx.GCDC(dc)

        # Setup
        dc.SetBrush(wx.TRANSPARENT_BRUSH)
        gc.SetBrush(wx.TRANSPARENT_BRUSH)
        gc.SetBackgroundMode(wx.TRANSPARENT)

        # The background needs some help to look transparent on
        # on Gtk and Windows
        if wx.Platform in ['__WXGTK__', '__WXMSW__']:
            gc.SetBackground(self.GetBackgroundBrush(gc))
            gc.Clear()

        # Get the working rectangle we can draw in
        rect = self.GetClientRect()

        rect.Inflate(-1, -1)

        # Draw rounded rectangle
        # Shadow
        gc.SetPen(wx.Pen(self.shadowColour, self.penThickness))
        gc.DrawRoundedRectangle(int(rect.x), int(rect.y),
                                int(rect.width), int(rect.height+1),
                                self.rounded)

        # Check if the widget is disabled  
        if not self.IsEnabled():
            # Set gray color for disabled state  
            color = wx.Colour(192, 192, 192, 100)  # Light gray  
            self.text_ctrl.SetForegroundColour(wx.Colour(130, 130, 130, 255))
        else:
            # Set green color for enabled state  
            color = wx.Colour(self.brushColour) #132, 178, 43, 100)  # Green
            self.text_ctrl.SetForegroundColour(self.textColour)
            
        # Draw rounded rectangle
        pen = wx.Pen(colour=self.penColour, width=self.penThickness)
        gc.SetPen(pen)        
        gc.SetBrush(wx.Brush(color))

        gc.DrawRoundedRectangle(int(rect.x), int(rect.y),
                                int(rect.width), int(rect.height),
                                self.rounded)


    def GetBackgroundBrush(self, dc):
        if wx.Platform == '__WXMAC__' or self._style:
            return wx.TRANSPARENT_BRUSH

        bkgrd = self.GetBackgroundColour()
        brush = wx.Brush(bkgrd, wx.SOLID)
        my_attr = self.GetDefaultAttributes()
        p_attr = self.Parent.GetDefaultAttributes()
        my_def = bkgrd == my_attr.colBg
        p_def = self.Parent.GetBackgroundColour() == p_attr.colBg
        if my_def and not p_def:
            bkgrd = self.Parent.GetBackgroundColour()
            brush = wx.Brush(bkgrd, wx.SOLID)
        return brush

    def SetWindowStyle(self, style):
        self._style = style
        self.Refresh()
        
    def OnEraseBackground(self, event):  
        pass
    
    def GetFontSizes(self):
        font = wx.SystemSettings.GetFont(wx.SYS_DEFAULT_GUI_FONT)
        font_sizes = []  # List to store font sizes

        if wx.Platform == '__WXMSW__':
            base_size = font.GetPointSize()
            font_sizes = [base_size, base_size + 1, base_size + 3, base_size + 6]  
        elif wx.Platform == '__WXGTK__':
            base_size = font.GetPointSize()
            font_sizes = [base_size, base_size, base_size + 2, base_size + 5]  
        elif wx.Platform == '__WXMAC__':
            base_size = font.GetPointSize()
            font_sizes = [base_size, base_size + 2, base_size + 6, base_size + 9]   
        else: 
            base_size = font.GetPointSize()
            font_sizes = [base_size]  # Default size

        return font_sizes  # Returns the list of font sizes

    def GetValue(self, event):
        return self.text_ctrl.GetValue()

    def SetValue(self, value):
        self.value = value  
        self.text_ctrl.SetValue(value)  # Updates the TextCtrl with the new value

