#!/usr/bin/env python
# -*- coding: utf-8 -*-
# py-indent-offset:4 -*-

#-------------------------------------------------------------------------------
# Name:             image_decoder_launcher.py
# Purpose:          Image decoder for images encoded with wx.tools and img2py
# Author:           Ecco
# Created:          2023
# Copyright:        ...
# License:          wxWindows license
# Version:          1.0.0
# Tags:             phoenix-port, py3-port
# ....
# Tested:           - Windows 10/11 | Python 3.11.9 |
# ....              wxPython 4.2.3 | wxWidgets 3.2.6
# ....
# ....              - Linux Mint 21 | Python 3.10.12 |
# ....              wxPython 4.2.1 gtk3 | wxWidgets 3.2.2.1
# ....
# ....              - MacOS Sequoia 15 | Python 3.12.4 |
# ....              wxPython 4.2.2 | wxWidgets 3.2.6
# ....
# Thanks to:        James Healey (Aka RolfofSaxony)
#-------------------------------------------------------------------------------

"""

1.0.0   First release

"""

#-------------------------------------------------------------------------------
# Import python packages
#-------------------------------------------------------------------------------
import sys
import os   
import platform
import importlib

#-------------------------------------------------------------------------------
# Import wxPython packages
#-------------------------------------------------------------------------------
import wx
from   stattext import GenStaticText
from   gradient_button import GradientButton

#-------------------------------------------------------------------------------
# Constant
#-------------------------------------------------------------------------------
STYLE_NOBG        = 90   # Useful on Windows to get a transparent appearance
                         # when the control is shown on a non solid background
                         
VERSION           = "1.0.0"

#-------------------------------------------------------------------------------
# Wildcard
#-------------------------------------------------------------------------------
# This is how you pre-establish a file filter so that the dialog
# only shows the extension(s) you want it to.
wildcard_1 = "Python source (*.py)|*.py"

wildcard_dict = {
    '.png': "Png files (*.png)|*.png",
    '.bmp': "Bmp files (*.bmp)|*.bmp",
    '.jpg': "Jpg files (*.jpg)|*.jpg",
    '.tif': "Tiff files (*.tif)|*.tif",
    '.ico': "Icon files (*.ico)|*.ico"
    #'.gif': "Gif files (*.gif)|*.gif",  don't works
}

#-------------------------------------------------------------------------------

# class BufferedPanel
# class Frame
# class App
# def main()

#-------------------------------------------------------------------------------

class BufferedPanel(wx.Panel):
    def __init__(self, parent,):
        wx.Panel.__init__(self, parent, style=wx.FULL_REPAINT_ON_RESIZE)

        if wx.Platform == "__WXMSW__":
            if not self.IsDoubleBuffered():
                self.SetDoubleBuffered(True)  # Reduce flicker on size event

#-------------------------------------------------------------------------------
    
class Frame(wx.Frame):
    def __init__(self, parent, id, title):
        style = wx.DEFAULT_FRAME_STYLE ^ (wx.MAXIMIZE_BOX|wx.RESIZE_BORDER|wx.STAY_ON_TOP)
        super().__init__(parent, -1, title, size=(-1, -1), style=style)

        # Return icons folder
        self.icons_dir = wx.GetApp().GetIconsDir()
        # Return source folder
        self.source_dir = wx.GetApp().GetSourceDir()
        # Return output folder
        self.output_dir = wx.GetApp().GetOutputDir()

        # State variables for buttons
        self.is_open_button_enabled = True
        self.is_save_button_enabled = False
        self.is_close_button_enabled = False

        # Variable to store the selected wildcard
        self.selected_wildcard = ""
        
        # Simplified init method
        self.SetProperties()
        self.CreateCtrls()
        self.BindEvents()
        self.DoLayout()

    def SetProperties(self):
        self.enableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                    "enabled_icon.ico"),
                                       type=wx.BITMAP_TYPE_ICO)
        self.SetIcon(self.enableFrameIcon)

        self.disableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                     "disabled_icon.ico"),
                                        type=wx.BITMAP_TYPE_ICO)

    def CreateCtrls(self):        
        # System font size
        font_sizes = self.GetFontSizes()

        selected_font_size = font_sizes[1]  # Select the second size from the list
        font = wx.SystemSettings.GetFont(wx.SYS_DEFAULT_GUI_FONT)
        font.SetPointSize(selected_font_size)  # Apply the selected size
        
        self.panel = BufferedPanel(self)
        self.panel.SetBackgroundColour("#545454")

        self.txt2 = GenStaticText(self.panel, -1, "Save images as :", style=STYLE_NOBG|wx.ALIGN_CENTER) 
        self.txt2.SetForegroundColour("#ffffff")
        self.txt2.SetFont(font)
        
        sampleList = ['.png', '.bmp', '.jpg', '.tif', '.ico'] #, 'ANY']  # '.gif' don't works
        self.choice = wx.Choice(self.panel, -1, choices=sampleList)
        self.choice.SetSelection(0)  
        self.choice.SetFocus()

        self.value = self.choice.GetString(0)  

        self.open_button = GradientButton(self.panel, id=-1, label="Open source folder", normalLabelColour="#ffffff",
                                    hoverLabelColour="#ffffff", pressedLabelColour="#ffffff",
                                    font_size=selected_font_size, font_weight=wx.FONTWEIGHT_NORMAL,
                                    normalBtnColour=(wx.Colour(146, 193, 48), wx.Colour(75, 120, 21)), 
                                    hoverBtnColour="#92c130",
                                    pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                                    #borderColour="#2574fe",
                                    focusColour="#ffffff",
                                    indicatorColour="#ffffff", shadowColour=wx.Colour(0, 0, 0, 0),
                                    focusSize=3, borderRadius=4, size=(-1, 30))
        self.open_button.SetToolTip("Select the folder containing the encoded images")
        self.open_button.Enable(self.is_open_button_enabled)

        
        self.save_button = GradientButton(self.panel, id=-1, label="Save decoded image", normalLabelColour="#ffffff",
                                    hoverLabelColour="#ffffff", pressedLabelColour="#ffffff",
                                    font_size=selected_font_size, font_weight=wx.FONTWEIGHT_NORMAL,
                                    normalBtnColour=(wx.Colour(146, 193, 48), wx.Colour(75, 120, 21)), 
                                    hoverBtnColour="#92c130",
                                    pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                                    #borderColour="#2574fe",
                                    focusColour="#ffffff",
                                    indicatorColour="#ffffff", shadowColour=wx.Colour(0, 0, 0, 0),
                                    focusSize=3, borderRadius=4, size=(-1, 35))
        self.save_button.SetToolTip("Save one or more encoded image(s) in png, \nbmp (without mask), jpg, ico\nor xpm format")   #  '.gif' don't works
        self.save_button.Enable(self.is_save_button_enabled)
        self.save_button.Enable(False)

        self.close_button = GradientButton(self.panel, id=-1, label="Close", normalLabelColour="#ffffff",
                                    hoverLabelColour="#ffffff", pressedLabelColour="#ffffff",
                                    font_size=selected_font_size, font_weight=wx.FONTWEIGHT_NORMAL,
                                    normalBtnColour=(wx.Colour(146, 193, 48), wx.Colour(75, 120, 21)), 
                                    hoverBtnColour="#92c130",
                                    pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                                    #borderColour="#2574fe",
                                    focusColour="#ffffff",
                                    indicatorColour="#ffffff", shadowColour=wx.Colour(0, 0, 0, 0),
                                    focusSize=3, borderRadius=4, size=(-1, 35))
        #self.close_button.SetToolTip(".....")
        
    def BindEvents(self):
        self.Bind(wx.EVT_ACTIVATE, self.OnActivate)
        self.Bind(wx.EVT_CHOICE, self.EvtChoice, self.choice)
        self.open_button.Bind(wx.EVT_BUTTON, self.OnOpenButton)
        self.save_button.Bind(wx.EVT_BUTTON, self.OnSaveButton)
        self.close_button.Bind(wx.EVT_BUTTON, self.OnClose)
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)

    def DoLayout(self):
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        btnSizer = wx.BoxSizer(wx.VERTICAL)
        borderSizer = wx.BoxSizer(wx.HORIZONTAL)
       
        btnSizer.Add(self.open_button, 1, wx.BOTTOM|wx.EXPAND, 10)
        btnSizer.Add(self.save_button, 1, wx.BOTTOM|wx.EXPAND, 10)
        btnSizer.Add(self.close_button, 1, wx.TOP|wx.EXPAND, 10)
        mainSizer.AddStretchSpacer()
        mainSizer.Add(self.txt2, 0, wx.BOTTOM, 5)   
        mainSizer.Add(self.choice, 1, wx.BOTTOM|wx.EXPAND, 10)
        mainSizer.Add(btnSizer, 1, wx.TOP|wx.EXPAND, 10)
        mainSizer.AddStretchSpacer()
        borderSizer.Add(mainSizer, 1, wx.ALL, 30)

        self.panel.SetSizer(borderSizer)
        self.Layout()

    def OnActivate(self, event):
        if event.GetActive():
            self.SetIcon(self.enableFrameIcon)
            self.panel.Enable(True)
            # Reactivate the buttons according to the previously stored states
            self.open_button.Enable(self.is_open_button_enabled)
            self.save_button.Enable(self.is_save_button_enabled)
            self.open_button.Enable(True)
        else:
            self.SetIcon(self.disableFrameIcon)
            self.panel.Enable(False)
            # Disable buttons
            self.is_open_button_enabled = self.open_button.IsEnabled()
            self.is_save_button_enabled = self.save_button.IsEnabled()
            self.save_button.Enable(False)
            self.open_button.Enable(False)

        self.Refresh()
        self.Layout()

    def GetFontSizes(self):
        font = wx.SystemSettings.GetFont(wx.SYS_DEFAULT_GUI_FONT)
        font_sizes = []  # List to store font sizes

        if wx.Platform == '__WXMSW__':
            base_size = font.GetPointSize()
            font_sizes = [base_size, base_size + 1]  
        elif wx.Platform == '__WXGTK__':
            base_size = font.GetPointSize()
            font_sizes = [base_size, base_size]  
        elif wx.Platform == '__WXMAC__':
            base_size = font.GetPointSize()
            font_sizes = [base_size, base_size + 3]   
        else:
            base_size = font.GetPointSize()
            font_sizes = [base_size]  # Default size

        return font_sizes  # Returns the list of font sizes
    
    def EvtChoice(self, event):
        selected_extension = event.GetString()
        print('EvtChoice: %s\n' % selected_extension)
        self.selected_wildcard = wildcard_dict.get(selected_extension, "")  # Get the wildcard
        print("Wildcard: ", self.selected_wildcard)
        self.value = selected_extension  
        
    def OnOpenButton(self, evt):
        print("CWD: %s\n" % os.getcwd())

        dlg = wx.FileDialog(
            self, message="Choose file(s)",
            defaultDir=self.source_dir,
            defaultFile="",
            wildcard=wildcard_1,
            style=wx.FD_OPEN | wx.FD_CHANGE_DIR | wx.FD_FILE_MUST_EXIST | wx.FD_PREVIEW)

        if dlg.ShowModal() == wx.ID_OK:
            # Get the path of the selected file  
            self.selected_files = dlg.GetPath()
            print("Name of selected file:", self.selected_files)

            # Extract the module name (without the extension)
            module_name = os.path.splitext(os.path.basename(self.selected_files))[0]
            module_dir = os.path.dirname(self.selected_files)  # Get the directory of the module

            # Add the module's directory to the sys.path  
            if module_dir not in sys.path:
                sys.path.append(module_dir)

            try:
                # Import module dynamically  
                imported_module = importlib.import_module(module_name)
                # Access catalog and index attributes 
                catalog = imported_module.catalog  
                index = imported_module.index  
                print("Module imported successfully!")

                # Create a list to store all the images to be backed up  
                self.images = []
                print("self.images", self.images) 

                # Browse the index to retrieve images  
                for img_id in index:
                    # Retrieve the image via the id  
                    image = catalog[img_id]

                    print("self.choice", self.value)
                    
                    # Add the image and its name in the desired format 
                    self.images.append((image.GetBitmap(), f"_{img_id}{self.value}"))

                    # Show image name (id)  
                    print(f"Image added: (_{img_id}.GetBitmap(), _{img_id}{self.value})")

                print(f"Total number of images recovered: {len(self.images)}")
                print("List of images:", self.images)  # View the list of images
                
            except ImportError as e:
                print(f"Error importing module: {e}")
            except AttributeError as e:
                print(f"Module does not have the required attributes: {e}")

        self.open_button.Enable(False)
        self.save_button.Enable(True)
        self.is_save_button_enabled = True

        dlg.Destroy()
    
    def OnSaveButton(self, evt):
        print("CWD: %s\n" % os.getcwd())
        
        # Check that the image list is not empty 
        if not self.images:
            wx.MessageBox("No images to save.", "Error", wx.OK | wx.ICON_ERROR)
            return

        # Use the first filename as the default  
        default_filename = self.images[0][1]
        print("Default file name:", default_filename)
        
        # Open dialog to choose output file 
        dlg = wx.FileDialog(
            self, message="Save file(s)...",
            defaultDir=self.output_dir,
            defaultFile=default_filename,
            wildcard=self.selected_wildcard,  
            style=wx.FD_SAVE | wx.FD_OVERWRITE_PROMPT  
        )

        if dlg.ShowModal() == wx.ID_OK:
            output_file = dlg.GetPath()
            print('You selected "%s"' % output_file)

            # Make sure you have defined 'images' somewhere  
            images = self.images 
            
            # Dictionary for mapping choices to bitmap types  
            bitmap_type_map = {
                '.png': wx.BITMAP_TYPE_PNG,
                '.bmp': wx.BITMAP_TYPE_BMP,
                '.jpg': wx.BITMAP_TYPE_JPEG,                
                '.tif': wx.BITMAP_TYPE_TIFF,
                '.ico': wx.BITMAP_TYPE_ICO,
                #'.gif': wx.BITMAP_TYPE_GIF,
                #'ANY': wx.BITMAP_TYPE_ANY,
            }
            print("bitmap_type_map", bitmap_type_map)

            selected_bitmap_type = getattr(self, 'value', self.choice.GetString(0))
            print("1 - selected_bitmap_type", selected_bitmap_type)
            bitmap_types = [bitmap_type_map[selected_bitmap_type]]  # Convert to constant wx
            print("2 - bitmap_types", bitmap_types)
            
            print("Available bitmap types:", bitmap_types)

            # Save each image 
            for bmp, filename in images:
                save_path = os.path.join(self.output_dir, filename)
                success = False

                # Try to save the bitmap with the selected format  
                for bitmap_type in bitmap_types:
                    if bmp.SaveFile(save_path, bitmap_type):
                        wx.MessageBox(f"Saved: {save_path} as {bitmap_type}", "Info",
                                  wx.OK | wx.ICON_INFORMATION)
                        success = True  
                        break  # Exit the loop if the save succeeds

                if not success:
                    wx.MessageBox(f"Failed to save: {save_path} in all formats.", "Error",
                                  wx.OK | wx.ICON_ERROR)

        self.save_button.Enable(False)
        dlg.Destroy()  # Make sure to destroy the dialogue at the end

    def OnClose(self, event):
        self.Close()

    def OnCloseWindow(self, event):
        self.Destroy()

#-------------------------------------------------------------------------------

class App(wx.App):
    def OnInit(self):
        
        self.installDir = os.path.split(os.path.abspath(sys.argv[0]))[0]

        # Gather system and library information  
        platform = wx.GetOsDescription()[:11]
        encoding = sys.getfilesystemencoding()
        python = sys.version.split()[0]
        wxpython = wx.VERSION_STRING[:5]
        wxwidgets = wx.GetLibraryVersionInfo().VersionString[10:]  # Slice to exclude the first 10 characters  
        name = "Image decoder"
        version = VERSION

        # Information to be printed  
        info = [
            ("Platform", platform),
            ("Encoding", encoding),
            ("Python", python),
            ("wxPython", wxpython),
            ("wxWidgets", wxwidgets),
            ("Name", name),
            ("Version", version)
        ]

        # Print the gathered information  
        print("")
        for label, value in info:
            print(f"{label} : {value}")
        print("")

        self.app_name = ("%s v%s" % (name, version))
        self.SetAppName(self.app_name)

        frame = Frame(None, -1, self.app_name)
        frame.SetSize((400, 300))
        self.SetTopWindow(frame)
        frame.Centre()
        frame.Show(True)

        return True

    def GetInstallDir(self):
        """
        Returns the installation directory for my application.
        """

        return self.installDir

    def GetSourceDir(self):
        """
        Returns the source directory for my application.
        """

        if not os.path.exists("source"):
            os.makedirs("source")
            
        source_dir = os.path.join(self.installDir, "source")
        return source_dir

    def GetOutputDir(self):
        """
        Returns the output directory for my application.
        """

        if not os.path.exists("output"):  
            os.makedirs("output")
            
        output_dir = os.path.join(self.installDir, "output")
        return output_dir

    def GetIconsDir(self):
        """
        Returns the icons directory for my application.
        """

        icons_dir = os.path.join(self.installDir, "icons")
        return icons_dir

    def GetVersion(self):
        """
        Return the current version for my application.
        """

        return __version__

#-------------------------------------------------------------------------------

def main():
    app = App(False)
    app.MainLoop()

#-------------------------------------------------------------------------------

if __name__ == "__main__":
    main()

