#!/usr/bin/env python
# -*- coding: utf-8 -*-
# py-indent-offset:4 -*-

#-------------------------------------------------------------------------------
# Name:             demo.py
# Purpose:          Test button
# Author:           Ecco
# Created:          2023
# Copyright:        ...
# License:          wxWindows license
# Version:          1.0.0
# Tags:             phoenix-port, py3-port
# ....
# Tested:           - Windows 10/11 | Python 3.11.9 |
# ....              wxPython 4.2.3 | wxWidgets 3.2.6
# ....
# ....              - Linux Mint 21 | Python 3.10.12 |
# ....              wxPython 4.2.1 gtk3 | wxWidgets 3.2.2.1
# ....
# ....              - MacOS Sequoia 15 | Python 3.12.4 |
# ....              wxPython 4.2.2 | wxWidgets 3.2.6
# ....
# Thanks to:        ...
#-------------------------------------------------------------------------------

"""
        
1.0.0   First release

"""

#-------------------------------------------------------------------------------
# Import python packages
#-------------------------------------------------------------------------------
import sys
import os
import array
import datetime      
import platform
import locale
import subprocess

#-------------------------------------------------------------------------------
# Import wxPython packages
#-------------------------------------------------------------------------------
import wx
import wx.lib.mixins.inspection  # Ctrl+Alt+I
from   button import Button
import wx.lib.colourdb
import wx.lib.fancytext as fancytext 
from   wx.html import HtmlWindow

#------------------------------------------------------------------------------- 
# Event ID'S       
#-------------------------------------------------------------------------------
ID_ABOUT         = wx.ID_ABOUT 
ID_BTN_CLOSE     = wx.NewIdRef() 
ID_EXIT          = wx.ID_EXIT     
ID_SYSTEM        = wx.NewIdRef()  
ID_THANKS_TO     = wx.NewIdRef()

ID_BITMAP_PDC    = wx.NewIdRef() 
ID_GRADIENT_PDC  = wx.NewIdRef()   
ID_COLOUR_PDC    = wx.NewIdRef()   
ID_SOLID_COLOUR  = wx.NewIdRef()  

ID_BTN_ONE       = wx.NewIdRef()  
ID_BTN_TWO       = wx.NewIdRef() 
ID_BTN_THREE     = wx.NewIdRef()  
ID_BTN_FOUR      = wx.NewIdRef() 
ID_BTN_FIVE      = wx.NewIdRef() 
ID_BTN_SIX       = wx.NewIdRef()
ID_BTN_SEVEN     = wx.NewIdRef()
ID_BTN_EIGHT     = wx.NewIdRef()
ID_BTN_NINE      = wx.NewIdRef()
ID_BTN_TEN       = wx.NewIdRef()
ID_BTN_ELEVEN    = wx.NewIdRef()
ID_BTN_TWELVE    = wx.NewIdRef()
ID_BTN_THIRTEEN  = wx.NewIdRef()
ID_BTN_FOURTEEN  = wx.NewIdRef()
ID_BTN_FIFTEEN  = wx.NewIdRef()

ID_BTN_FIFTY_ONE  = wx.NewIdRef()
ID_BTN_FIFTY_TWO  = wx.NewIdRef()
ID_BTN_FIFTY_TREE = wx.NewIdRef()
ID_BTN_FIFTY_FOUR = wx.NewIdRef()
ID_BTN_FIFTY_FIVE = wx.NewIdRef()

#-------------------------------------------------------------------------------
# Constant
#-------------------------------------------------------------------------------
SHOW_BACKGROUND   = 1

BTN_RND_RECTANGLE = 20   # Define new appearance constants
BTN_RECTANGLE     = 21

VERSION           = "1.0.0"
LICENSE           = "wxWindows license"
THANKS_MESSAGE    = ["Special thanks to :",
                     "",
                     "Cody Precord",
                     ]
MACOS_MESSAGE = """To ensure proper control tab navigation on MacOS, please adjust the following settings:

Apple Menu / System Settings... / Accessibility / Keyboard / Full Keyboard Access / Toggle switch On

or

Apple Menu / System Settings... / Keyboard / Keyboard Navigation / Toggle switch On"""

#-------------------------------------------------------------------------------

# class ScrolledPanel
# class BufferedPanel
# class ThanksToPanel
# class ThanksToDialog 
# class SystemDialog
# class AboutDialog  
# class BitmapPanelPdc
# class GradientPanelPdc
# class ColourPanelPdc
# class ColourPanel
# class DefaultPanel
# class MainPanel     
# class Frame
# class App
# def main

#-------------------------------------------------------------------------------

class ScrolledPanel(wx.ScrolledWindow):
    def __init__(self, parent):
        super(ScrolledPanel, self).__init__(parent, style=wx.TAB_TRAVERSAL |
                                            wx.FULL_REPAINT_ON_RESIZE)

        # Attributes
        self.parent = parent
        
        if wx.Platform == "__WXMSW__":
            if not self.IsDoubleBuffered():
                self.SetDoubleBuffered(True)  # Reduce flicker on size event

        self.SetScrollRate(5,5)
        
#-------------------------------------------------------------------------------

class BufferedPanel(wx.Panel):
    def __init__(self, parent):
        super(BufferedPanel, self).__init__(parent, style=wx.TAB_TRAVERSAL |
                                            wx.FULL_REPAINT_ON_RESIZE)
        
        if wx.Platform == "__WXMSW__":
            if not self.IsDoubleBuffered():
                self.SetDoubleBuffered(True)  # Reduce flicker on size event

#-------------------------------------------------------------------------------

class ThanksToPanel(wx.Panel):   
    def __init__(self, parent, message,
                 backgroundColour=(wx.Colour(0, 0, 0, 255)),
                 fading=255, image_path=None, size=(100, 100)):
        super(ThanksToPanel, self).__init__(parent, -1, size=size)

        # Attributes      
        self.parent = parent
        self.message = message
        self.backgroundColour = backgroundColour
        self.fading = fading
        w, h = self.GetClientSize()
        self.scroll_speed = 0.5      # Text scrolling speed
        self.height = 50             # 50 --> 1 (without fade-out)
        self.y = h                   # Initial y position of image
        self.photo = None            # Initialize the image to None
        
        self.timer = wx.Timer(self)
        # Start(int milliseconds, oneShot=False)
        # Lower timer interval gives faster speed
        self.timer.Start(10) 

        # Return bitmaps folder
        self.bitmaps_dir = wx.GetApp().GetBitmapsDir()

        self.MakeBitmapRGBA(w+100, self.height, col_offset=True)   # width, height for gradient 
        self.MakeBitmapRGBA(w+100, self.height, col_offset=False)

        # Simplified init method
        self.SetProperties()
        self.CreateCtrls(image_path)
        self.BindEvents()

    def SetProperties(self):
        self.SetBackgroundStyle(wx.BG_STYLE_CUSTOM)

    def CreateCtrls(self, image_path):
        if image_path:
            self.photo = wx.Bitmap(os.path.join(self.bitmaps_dir, image_path),
                                   type=wx.BITMAP_TYPE_ANY)  
        else:
            self.photo = None 
        
    def BindEvents(self):      
        self.Bind(wx.EVT_SIZE, self.OnResize)
        self.Bind(wx.EVT_PAINT, self.OnPaint)
        self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnErase)        
        self.Bind(wx.EVT_TIMER, self.OnTimer, self.timer)     
        self.Bind(wx.EVT_KEY_DOWN, self.OnKeyDown)
        self.Bind(wx.EVT_KEY_UP, self.OnKeyUp)

    def OnKeyDown(self, event):
        hasFocus = self.HasFocus()
        if hasFocus and event.GetKeyCode() == wx.WXK_SPACE:
            self.timer.Stop()
        event.Skip()

    def OnKeyUp(self, event):
        hasFocus = self.HasFocus()
        if hasFocus and event.GetKeyCode() == wx.WXK_SPACE:
            self.timer.Start() 
        event.Skip()
        
    def OnErase(self, event):
        pass
           
    def OnResize(self, event):
        self.Refresh()
        event.Skip()        
      
    def GetRGB(self, x, y, bpp):   
        # Calculate some colour values for
        # this sample based on x,y position
        # 255 = white  |  0 = black  (fading)
        r = g = b = self.fading

        if bpp == 4:   
            a = int(y * 255.0 / self.height)  

            return r, g, b, a   
        else:   
            return r, g, b      
       
    def MakeBitmapRGBA(self, width, height, col_offset):   
        # Make a bitmap using an array of RGBA bytes 
        bpp = 4  # Bytes per pixel  
        bytes = array.array('B', [0] * width * height * bpp)   
   
        for y in range(height):   
            for x in range(width):   
                if col_offset:
                    offset = x * bpp - y * width * bpp  
                else:
                    offset = y * width * bpp - x * bpp
                
                r, g, b, a = self.GetRGB(x, y, bpp)   
                bytes[offset + 0] = r   
                bytes[offset + 1] = g   
                bytes[offset + 2] = b   
                bytes[offset + 3] = a   
   
        if col_offset:
            self.rgbaBmp = wx.Bitmap.FromBufferRGBA(width, height, bytes)
        else:
            self.rgbaBmp_ = wx.Bitmap.FromBufferRGBA(width, height, bytes)   
    
    def DrawBitmapAndMessage(self, dc, bmp, msg, x_, y_):   
        x, y = x_, y_   
   
        dc.DrawText(msg, x, y)   

        # Draw the bitmap over the text   
        dc.DrawBitmap(bmp, x_, y_, True)

    def CalculateTotalTextHeight(self):
        # System font
        ft_size = self.GetTextFontSize()
        
        font = wx.Font(ft_size, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD)
        dc = wx.ClientDC(self)
        dc.SetFont(font)
        
        line_height = dc.GetTextExtent("A")[1]  # Use only one letter to get line height

        # Count the number of lines (ignoring empty lines)
        num_lines = sum(1 for msg in self.message if msg.strip() != "")
    
        return num_lines * line_height + 80  # Add offset for first message and add padding for additional aesthetics

    def OnPaint(self, event):   
        dc = wx.BufferedPaintDC(self) 
        brush = wx.Brush(self.backgroundColour)     
        dc.SetBackground(brush)
        dc.Clear()

        if self.photo:  # Check if the image has been loaded   
            w = self.photo.GetWidth()
            h = self.photo.GetHeight()
            width, height = self.GetSize()
            l, h = ((width / 2) - (w / 2), (height / 2) - (h / 2))

            # Draw the image
            dc.DrawBitmap(self.photo, int(l), int(h), False)

        self.PaintBox(dc)
             
    def PaintBox(self, dc):
        # System font
        ft_size = self.GetTextFontSize()
        
        font = wx.Font(ft_size, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD)
        dc.SetFont(font)
        
        # One colour for the set
        #colours = ["#ffffff"]*19  # white
        
        # Or one colour per segment
        colours = [
            "#ffffff", "",
            #"#00ffc8",
            #"#00ff11", "#8cff00", "#d9ff00", 
            #"#ffe500",
            "#ffc800", "#ff7f00", "#ff5100",
        ]
        
        for idx, (msg, colour) in enumerate(zip(self.message, colours)):
            dc.SetTextForeground(colour)
            y_position = int(self.y + 20 + idx * 20)
            dc.DrawText(msg, int(self.GetSize().width / 2 - dc.GetTextExtent(msg)[0] / 2), y_position)

            # Insert line after first message  
            if idx == 0:  # If this is the first message ("Special thanks to:")
                dc.SetPen(wx.Pen('#787878', 2))
                # Draw the line  
                width = self.GetSize().width  
                dc.DrawLine(100, y_position + 30, width - 100, y_position + 30)

        self.DrawBitmapAndMessage(dc, self.rgbaBmp, "", 0 - 100, -1)                                                                        
        self.DrawBitmapAndMessage(dc, self.rgbaBmp_, "", 0 - 100,
                                  self.GetSize().height - 50 + 1)  # 50 --> 1 (without fade-in)

    def GetTextFontSize(self):
        font = wx.SystemSettings.GetFont(wx.SYS_DEFAULT_GUI_FONT)
        font_size = font.GetPointSize()
    
        # Dictionary for size adjustments by platform  
        platform_adjustments = {
            '__WXMSW__': 1,
            '__WXGTK__': -1,
            '__WXMAC__': 2  
        }
    
        # Use get() to get the fit, or 0 if the platform is not specified
        adjustment = platform_adjustments.get(wx.Platform, 0)
  
        return font_size + adjustment  # Returns the adjusted font size

    def SetScrollSpeed(self, speed):
        self.scroll_speed = speed
        
    def OnTimer(self, event):
        w, h = self.GetClientSize()

        # Increment x to start moving using the scroll speed  
        self.y += -self.scroll_speed  # Using scroll speed

        # Calculation of total height of posts  
        total_text_height = self.CalculateTotalTextHeight()

        # Limit movement based on total text height
        if self.y > h or self.y < -total_text_height-60:
            self.y = h  
        
        self.Refresh()

    def OnCleanup(self):
        self.timer.Stop()
        
#-------------------------------------------------------------------------------

class ThanksToDialog(wx.Dialog):
    def __init__(self, parent, id, message, backgroundColour, fading,
                 title, size=wx.DefaultSize, pos=wx.DefaultPosition,
                 style=wx.DEFAULT_DIALOG_STYLE, name='dialog', image_path=None):
        super(ThanksToDialog, self).__init__(parent, id, title, pos, size, style, name)
        
        # Attributes
        self.parent = parent
        self.message = message
        self.backgroundColour = backgroundColour
        self.fading = fading
        self.image_path = image_path  # Store the image path 
        self.ticker = None   # Store the reference to ThanksToPanel
        
        # Return icons folder
        self.icons_dir = wx.GetApp().GetIconsDir()
        
        # Simplified init method
        self.SetProperties()
        self.CreateCtrls()
        self.BindEvents()
        self.DoLayout()
        
    def SetProperties(self):
        self.enableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                    "enabled_icon.ico"),
                                       type=wx.BITMAP_TYPE_ICO)
        self.SetIcon(self.enableFrameIcon)

        self.disableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                     "disabled_icon.ico"),
                                        type=wx.BITMAP_TYPE_ICO) 

        self.SetTransparent(240)
        self.SetSize((360, 420))
        
    def CreateCtrls(self):
        self.panel1 = wx.Panel(self,
                               id=-1,
                               style=wx.BORDER_NONE |
                               wx.TAB_TRAVERSAL)
        self.panel1.SetBackgroundColour(self.backgroundColour)

        self.panel2 = wx.Panel(self,
                               id=-1,
                               style=wx.TAB_TRAVERSAL)

        w, h = self.GetClientSize()                 

        self.ticker = ThanksToPanel(self.panel1, message=self.message,
                                    backgroundColour=self.backgroundColour,
                                    fading=self.fading, 
                                    image_path=self.image_path, size=(w, h))
        # if wx.Platform in ['__WXMAC__']:
        #     self.ticker.SetScrollSpeed(0.3)
        # else:
        #     self.ticker.SetScrollSpeed(0.5)

        #------------
     
        self.btnClose = wx.Button(self.panel2,
                                  id=wx.ID_CLOSE,
                                  label="&Close")
        self.btnClose.SetFocus()       
    
    def BindEvents(self):
        self.Bind(wx.EVT_ACTIVATE, self.OnActivate)
        self.Bind(wx.EVT_BUTTON, self.OnClose, id=wx.ID_CLOSE)
        self.Bind(wx.EVT_CHAR_HOOK, self.OnKeyUp)
        self.Bind(wx.EVT_CLOSE, self.OnClose) 
        
    def DoLayout(self):
        hBox1 = wx.BoxSizer(wx.HORIZONTAL)
        hBox1.Add(self.ticker, 1, wx.ALL | wx.EXPAND, 5)
        self.panel1.SetSizer(hBox1)

        hBox2 = wx.BoxSizer(wx.HORIZONTAL)
        hBox2.Add(self.btnClose, 1, wx.BOTTOM | wx.EXPAND, 5)
        self.panel2.SetSizer(hBox2)

        vBox = wx.BoxSizer(wx.VERTICAL)
        vBox.Add(self.panel1, 1, wx.EXPAND | wx.ALL | wx.CENTER, 5)    
        vBox.Add(self.panel2, 0, wx.ALL | wx.CENTER,  5)    

        self.SetAutoLayout(True)
        self.SetSizer(vBox)

    def OnActivate(self, event):
        if event.GetActive():
            self.SetIcon(self.enableFrameIcon)
            self.btnClose.Enable(True)
        else:
            self.SetIcon(self.disableFrameIcon)
            self.btnClose.Enable(False)

        self.Refresh()  # Redraw the window to reflect the changes

    def SetScrollSpeed(self, speed):
        if self.ticker:
            self.ticker.SetScrollSpeed(speed)
            
    def OnKeyUp(self, event):
        if event.GetKeyCode() == wx.WXK_ESCAPE:
            # Close the dialog, no action
            self.OnClose(event)
        event.Skip()

    def OnClose(self, event):
        self.ticker.OnCleanup()  # Stop the timer before closing
        self.EndModal(0) 
        
#-------------------------------------------------------------------------------
                
class SystemDialog(wx.Dialog):  
    def __init__(self, parent, id, title, size=wx.DefaultSize, pos=wx.DefaultPosition,
                 style=wx.DEFAULT_DIALOG_STYLE, name='dialog'):
        super(SystemDialog, self).__init__(parent, id, title, pos, size, style, name) 
        
        # Colourdb  
        wx.lib.colourdb.updateColourDB()

        # Attributes
        self.active = True
        self.SetTransparent(255)
        self.parent = parent
          
        # Return icons folder  
        self.icons_dir = wx.GetApp().GetIconsDir()        
        # Return html folder 
        self.html_dir = wx.GetApp().GetHtmlDir()
    
        # Simplified init method
        self.SetProperties()
        self.CreateCtrls()
        self.BindEvents()
        self.DoLayout()

    def SetProperties(self):
        self.enableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                    "enabled_icon.ico"),
                                       type=wx.BITMAP_TYPE_ICO)
        self.SetIcon(self.enableFrameIcon)

        self.disableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                     "disabled_icon.ico"),
                                        type=wx.BITMAP_TYPE_ICO)           

    def CreateCtrls(self):
        self.html = HtmlWindow(self, id=-1, size=(300, 225), 
                               style=wx.html.HW_NO_SELECTION|
                                     wx.BORDER_SIMPLE)

        if wx.Platform == '__WXGTK__':
            font_size = 11 
        elif wx.Platform == '__WXMSW__':
            font_size = 10  
        elif wx.Platform == '__WXMAC__':
            font_size = 14  
        else:
            font_size = 11  

        self.html.SetStandardFonts(font_size, "")

        self.LoadActiveHtmlContent()  # Load active HTML content initially

        #------------

        self.ok_button = wx.Button(self, id=wx.ID_OK, label="&OK")
        self.ok_button.SetFocus()

    def BindEvents(self):
        self.Bind(wx.EVT_ACTIVATE, self.OnActivate) 
        self.ok_button.Bind(wx.EVT_BUTTON, self.OnClose)
        self.Bind(wx.EVT_CHAR_HOOK, self.OnKeyUp)
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)

    def DoLayout(self):
        # main_sizer is the top-level one that manages everything
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        #------------

        # Assign widgets to sizers
        # wx.BoxSizer(window, proportion, flag, border)
        # wx.BoxSizer(sizer, proportion, flag, border)
        main_sizer.Add(self.html,
                       proportion=1,
                       flag=wx.TOP | wx.LEFT | wx.RIGHT | wx.EXPAND,
                       border=5)

        main_sizer.Add(self.ok_button,
                       proportion=0,
                       flag=wx.ALIGN_CENTER | wx.ALL,
                       border=10)

        self.SetSizer(main_sizer)
        main_sizer.Fit(self)
        
    def LoadActiveHtmlContent(self):
        # Get OS information  
        os_name                = platform.system()         
        os_release, os_version = self.GetOsInfo(os_name)
        
        python                 = sys.version.split()[0]
        wx_python              = wx.VERSION_STRING[:5]
        wxWidgets              = wx.GetLibraryVersionInfo().VersionString
        wx_widgets             = wxWidgets.split()[1]
        encoding               = sys.getfilesystemencoding()
        # coding               = wx.PlatformInfo[2]  # unicode (coding)
 
        print(f"OS name: {os_name}")               # OS name: Windows
        print(f"OS release: {os_release}")         # OS release: Windows 10
        print(f"OS version: {os_version}")         # OS version: 10.0.19045
        print(f"Python version: {python}")         # Python version: 3.11.9
        print(f"wxPython version: {wx_python}")    # wxPython version: 4.2.3       
        print(f"wxWidgets version: {wx_widgets}")  # wxWidgets version: 3.2.6
        print(f"Encoding: {encoding}\n")           # Encoding: Utf-8
        
        #------------

        # Specify the HTML files for active states  
        self.html_file = os.path.join(self.html_dir, "active_system.html")
        
        with open(self.html_file, "r") as myTemplate:
            strTemplate = myTemplate.read()
            self.sample_html = strTemplate
            self.sample_html = self.sample_html.replace("{{OS}}", os_name)
            self.sample_html = self.sample_html.replace("{{DISTRIBUTION}}", os_release)
            self.sample_html = self.sample_html.replace("{{VERSION}}", os_version)
            self.sample_html = self.sample_html.replace("{{PYTHON}}", python)
            self.sample_html = self.sample_html.replace("{{WXPYTHON}}", wx_python)
            self.sample_html = self.sample_html.replace("{{WXWIDGETS}}", wx_widgets)
            self.sample_html = self.sample_html.replace("{{ENCODING}}", encoding.capitalize())
            # self.sample_html = self.sample_html.replace("{{UNICODE}}", unicode)
            
        self.html.SetPage(self.sample_html)

    def LoadInactiveHtmlContent(self):
        # Get OS information  
        os_name                = platform.system()         
        os_release, os_version = self.GetOsInfo(os_name)
        
        python                 = sys.version.split()[0]
        wx_python              = wx.VERSION_STRING[:5]
        wxWidgets              = wx.GetLibraryVersionInfo().VersionString
        wx_widgets             = wxWidgets.split()[1]
        encoding               = sys.getfilesystemencoding()
        # coding               = wx.PlatformInfo[2]  # unicode (coding)
        
        #------------

        # Specify the HTML files for active states  
        self.html_file = os.path.join(self.html_dir, "inactive_system.html")
        
        with open(self.html_file, "r") as myTemplate:
            strTemplate = myTemplate.read()
            self.sample_html = strTemplate
            self.sample_html = self.sample_html.replace("{{OS}}", os_name)
            self.sample_html = self.sample_html.replace("{{DISTRIBUTION}}", os_release)
            self.sample_html = self.sample_html.replace("{{VERSION}}", os_version)
            self.sample_html = self.sample_html.replace("{{PYTHON}}", python)
            self.sample_html = self.sample_html.replace("{{WXPYTHON}}", wx_python)
            self.sample_html = self.sample_html.replace("{{WXWIDGETS}}", wx_widgets)
            self.sample_html = self.sample_html.replace("{{ENCODING}}", encoding.capitalize())
            # self.sample_html = self.sample_html.replace("{{UNICODE}}", unicode)
            
        self.html.SetPage(self.sample_html)

    def GetOsInfo(self, os_name):
        if os_name == "Linux":
            return self.GetLinuxInfo()
        elif os_name == "Darwin":
            return self.GetMacosInfo()
        elif os_name == "Windows":
            return self.GetWindowsInfo()
        else:
            return "N/A", platform.version()

    def GetLinuxInfo(self):
        try:
            import distro  # Make sure the 'distro' module is installed for Linux  
            #distribution = distro.linux_distribution()
            #os_release = f"{distribution[0]} {distribution[1]}"
            #os_version = platform.release()
            os_release = distro.name()
            os_version = distro.version()
        except ImportError:
            os_release = "Linux"
            os_version = platform.release()        
        return os_release, os_version

    def GetMacosInfo(self):
        os_release = wx.GetOsDescription()
        segments = os_release.split(' ')[:2]
        os_release = ' '.join(segments)
        os_version = platform.mac_ver()[0] 
        return os_release.title(), os_version

    def GetWindowsInfo(self):
        os_release = f"Windows {platform.release()}"
        os_version = platform.version()
        return os_release, os_version
                
    def OnActivate(self, event):
        self.active = event.GetActive()
        if self.active:
            self.SetIcon(self.enableFrameIcon)
            self.LoadActiveHtmlContent()  # Load active HTML content  
            self.ok_button.Enable(True)
        else:
            self.SetIcon(self.disableFrameIcon)
            self.LoadInactiveHtmlContent()  # Load inactive HTML content  
            self.ok_button.Enable(False)
        self.Refresh()  # Redraw the window to reflect the changes

    def OnKeyUp(self, event):
        if event.GetKeyCode() == wx.WXK_ESCAPE:
            # Close the frame, no action  
            self.OnCloseWindow(event)
        event.Skip()

    def OnClose(self, event):
        self.Close()

    def OnCloseWindow(self, event):
        self.EndModal(0)
     
#-------------------------------------------------------------------------------
        
class AboutDialog(wx.Dialog):
    def __init__(self, parent, id, title, size=wx.DefaultSize, pos=wx.DefaultPosition,
                 style=wx.DEFAULT_DIALOG_STYLE, name='dialog'):
        super(AboutDialog, self).__init__(parent, id, title, pos, size, style, name)
        
        # Colourdb
        wx.lib.colourdb.updateColourDB()
        
        # Attributes
        self.parent = parent
        self.delta = wx.Point(0,0)
        
        if wx.Platform == "__WXMSW__":
            self.SetTransparent(240)
        elif wx.Platform == "__WXMAC__":
            self.SetTransparent(240)
        else:
            pass
        
        # Return bitmaps folder
        self.bitmaps_dir = wx.GetApp().GetBitmapsDir()
        # Return icons folder
        self.icons_dir = wx.GetApp().GetIconsDir()
        
        # Simplified init method
        self.SetProperties()
        self.CreateCtrls()
        self.BindEvents()
        self.DoLayout()
        
    def SetProperties(self):
        self.enableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                    "enabled_icon.ico"),
                                       type=wx.BITMAP_TYPE_ICO)
        self.SetIcon(self.enableFrameIcon)

        self.disableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                     "disabled_icon.ico"),
                                        type=wx.BITMAP_TYPE_ICO)       

    def CreateCtrls(self):
        self.panel = BufferedPanel(self)

        # A font can be retrieved from the OS 
        # default font and modified
        font = wx.SystemSettings.GetFont(wx.SYS_DEFAULT_GUI_FONT)
        font.SetStyle(wx.NORMAL)
        font.SetWeight(wx.BOLD)
        
        # Return bitmaps folder
        self.bitmaps_dir = wx.GetApp().GetBitmapsDir()

        #------------

        self.bitmap = wx.Bitmap(os.path.join(self.bitmaps_dir,
                                             "bmp_about.png"),
                                    type=wx.BITMAP_TYPE_PNG)

        # Convert bitmap to grayscale 
        gray_bitmap = self.ConvertToGrayscale(self.bitmap)
        
        # Create a bitmap from the modified image  
        self.gray_bitmap = wx.Bitmap(gray_bitmap)
        self.logo = wx.StaticBitmap(self.panel, bitmap=self.gray_bitmap )

        #------------
        
        # Set frame title
        name = "%s" % (wx.GetApp().GetAppName())
        space = "%s" % ("  ")
        appName = ("{}{}{}".format(space, name, space))

        self.title = wx.StaticText(self.panel,
                                   -1,
                                   appName,
                                   size=(270, -1),
                                   style=wx.ALIGN_CENTER)
        self.title.SetForegroundColour("grey25")
        
        #------------

        self.btn1 = wx.Button(self.panel,
                              ID_BTN_FIFTY_ONE,
                              size=(160, 22))
        self.btn1.SetLabelMarkup(''.join(('<b>', '&Version', '</b>')))

        self.sz1 = BufferedPanel(self.panel)

        bsizer1 = wx.BoxSizer(wx.VERTICAL)

        self.sz1.Show(not self.sz1.IsShown())

        self.txt1 = wx.StaticText(self.sz1,
                                  -1,
                                  ("v" + VERSION),
                                  style=wx.ALIGN_CENTER)
        self.txt1.SetForegroundColour("grey25")
        
        bsizer1.Add(self.txt1, 0, wx.BOTTOM|wx.LEFT|wx.RIGHT|wx.ALL, 5)
        self.sz1.SetSizer(bsizer1)

        #------------

        self.btn2 = wx.Button(self.panel,
                              ID_BTN_FIFTY_TWO,
                              size=(160, 22))
        self.btn2.SetLabelMarkup(''.join(('<b>', '&Developer(s)', '</b>')))

        self.sz2 = BufferedPanel(self.panel)

        bsizer2 = wx.BoxSizer(wx.VERTICAL)

        self.sz2.Show(not self.sz2.IsShown())

        self.txt2 = wx.StaticText(self.sz2,
                                  -1,
                                  "Ecco",
                                  style=wx.ALIGN_CENTER)
        self.txt2.SetForegroundColour("grey25")

        bsizer2.Add(self.txt2, 0, wx.BOTTOM|wx.LEFT|wx.RIGHT|wx.ALL, 5)
        self.sz2.SetSizer(bsizer2)

        #------------

        self.btn3 = wx.Button(self.panel,
                              ID_BTN_FIFTY_TREE,
                              size=(160, 22))
        self.btn3.SetLabelMarkup(''.join(('<b>', '&Thanks to...', '</b>')))

        self.sz3 = BufferedPanel(self.panel)

        bsizer3 = wx.BoxSizer(wx.VERTICAL)

        self.sz3.Show(not self.sz3.IsShown())

        self.txt3 = wx.StaticText(self.sz3,
                                  -1,
                                  "Read source or menu",
                                  style=wx.ALIGN_CENTER)
        self.txt3.SetForegroundColour("grey25")
        
        bsizer3.Add(self.txt3, 0, wx.BOTTOM|wx.LEFT|wx.RIGHT|wx.ALL, 5)
        self.sz3.SetSizer(bsizer3)

        #------------
        
        self.btn4 = wx.Button(self.panel,
                              ID_BTN_FIFTY_FOUR,
                              size=(160, 22))
        self.btn4.SetLabelMarkup(''.join(('<b>', '&License', '</b>')))

        self.sz4 = BufferedPanel(self.panel)

        bsizer4 = wx.BoxSizer(wx.VERTICAL)

        self.sz4.Show(not self.sz4.IsShown())

        self.txt4 = wx.StaticText(self.sz4,
                                  -1,
                                  LICENSE,
                                  style=wx.ALIGN_CENTER)
        self.txt4.SetForegroundColour("grey25")
        
        bsizer4.Add(self.txt4, 0, wx.BOTTOM|wx.LEFT|wx.RIGHT|wx.ALL, 5)
        self.sz4.SetSizer(bsizer4)

        #------------
        self.btn5 = wx.Button(self.panel,
                              ID_BTN_FIFTY_FIVE,
                              size=(160, 22))
        self.btn5.SetLabelMarkup(''.join(('<b>', '&Copyright', '</b>')))

        self.sz5 = BufferedPanel(self.panel)

        bsizer5 = wx.BoxSizer(wx.VERTICAL)

        self.sz5.Show(not self.sz5.IsShown())

        self.txt5 = wx.StaticText(self.sz5,
                                  -1,
                                  "...",   # (c) 2023
                                  style=wx.ALIGN_CENTER)
        self.txt5.SetForegroundColour("grey25")
        
        bsizer5.Add(self.txt5, 0, wx.BOTTOM|wx.LEFT|wx.RIGHT|wx.ALL, 5)
        self.sz5.SetSizer(bsizer5)

        #------------

        self.line = wx.StaticLine(self.panel, -1, size=(100, -1), style=wx.LI_HORIZONTAL)

        self.btn6 = wx.Button(self.panel,
                              wx.ID_OK,
                              size=(160, -1))
        self.btn6.SetLabelMarkup(''.join(('<b>', '&OK', '</b>')))
        self.btn6.SetForegroundColour("blue")
        self.btn6.SetFont(font)
            
    def BindEvents(self):
        self.Bind(wx.EVT_ACTIVATE, self.OnActivate)
        self.btn1.Bind(wx.EVT_BUTTON, self.OnButton)
        self.btn2.Bind(wx.EVT_BUTTON, self.OnButton)
        self.btn3.Bind(wx.EVT_BUTTON, self.OnButton)
        self.btn4.Bind(wx.EVT_BUTTON, self.OnButton)
        self.btn5.Bind(wx.EVT_BUTTON, self.OnButton)
        self.btn6.Bind(wx.EVT_BUTTON, self.OnButton)
        
        self.Bind(wx.EVT_CHAR_HOOK, self.OnKeyUp)
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)
        self.Bind(wx.EVT_LEFT_UP, self.OnLeftUp)
        self.Bind(wx.EVT_MOTION, self.OnMouseMove)

        self.panel.Bind(wx.EVT_LEFT_DOWN, self.OnLeftDown)      
        self.panel.Bind(wx.EVT_RIGHT_UP, self.OnCloseWindow)  

    def DoLayout(self):
        self.sizer = wx.BoxSizer(wx.VERTICAL)
        self.sizer.Add(self.btn1, proportion=0, flag=wx.EXPAND|wx.CENTRE|wx.LEFT|wx.RIGHT, border=20)
        self.sizer.Add(self.btn2, proportion=0, flag=wx.EXPAND|wx.CENTRE|wx.LEFT|wx.RIGHT, border=20)
        self.sizer.Add(self.btn3, proportion=0, flag=wx.EXPAND|wx.CENTRE|wx.LEFT|wx.RIGHT, border=20)
        self.sizer.Add(self.btn4, proportion=0, flag=wx.EXPAND|wx.CENTRE|wx.LEFT|wx.RIGHT, border=20)
        self.sizer.Add(self.btn5, proportion=0, flag=wx.EXPAND|wx.CENTRE|wx.LEFT|wx.RIGHT, border=20)

        self.btnSizer = wx.BoxSizer(wx.HORIZONTAL)
        self.btnSizer.Add(self.sizer, proportion=1, flag=wx.CENTRE)

        self.btnOkSizer = wx.BoxSizer(wx.HORIZONTAL)
        self.btnOkSizer.Add(self.btn6, proportion=1, flag=wx.EXPAND|wx.CENTRE|wx.LEFT|wx.RIGHT, border=20)
        
        self.panelSizer = wx.StaticBoxSizer(wx.StaticBox(self, -1, ""), wx.VERTICAL)

        self.panelSizer.Add(self.logo, proportion=1, flag=wx.CENTRE)
        self.panelSizer.Add((0, 10), 0)
        self.panelSizer.Add(self.title, proportion=0, flag=wx.CENTRE)
        self.panelSizer.Add((0, 10), 0)
        self.panelSizer.Add(self.btnSizer, proportion=0, flag=wx.EXPAND|wx.CENTRE)
        self.panelSizer.Add(self.sz1, proportion=0, flag=wx.CENTRE)
        self.panelSizer.Add(self.sz2, proportion=0, flag=wx.CENTRE)
        self.panelSizer.Add(self.sz3, proportion=0, flag=wx.CENTRE)
        self.panelSizer.Add(self.sz4, proportion=0, flag=wx.CENTRE)
        self.panelSizer.Add(self.sz5, proportion=0, flag=wx.CENTRE)
        self.panelSizer.Add((0, 0), 0, wx.EXPAND)
        self.panelSizer.Add(self.line, 0, wx.CENTRE|wx.TOP, 10)
        self.panelSizer.Add((0, 0), 0, wx.EXPAND)
        self.panelSizer.Add(self.btnOkSizer, 0, wx.EXPAND|wx.CENTRE | wx.TOP | wx.BOTTOM, 10)

        self.panel.SetSizer(self.panelSizer)
        
        #------------

        mainSizer = wx.BoxSizer()
        mainSizer.Add(self.panel, proportion=1, flag=wx.EXPAND)

        #------------

        self.SetSizer(mainSizer)
        self.Layout()
        self.Fit()

    def ConvertToGrayscale(self, bitmap):
        """
        Convert the given bitmap to a grayscale bitmap.
        """
        
        # Convert bitmap to image  
        image = bitmap.ConvertToImage()
    
        # Set a lightening factor  
        brighten_factor = 10  # You can adjust this value according to your needs
    
        # Convert image to grayscale 
        for x in range(image.GetWidth()):
            for y in range(image.GetHeight()):
                r = image.GetRed(x, y)
                g = image.GetGreen(x, y)
                b = image.GetBlue(x, y)
                gray = int(0.299 * r + 0.587 * g + 0.114 * b)  # Brightness method
            
                # Add the lightening factor while making sure not to exceed 255  
                gray = min(255, gray + brighten_factor)
            
                image.SetRGB(x, y, gray, gray, gray)

        # Create a bitmap from the edited image  
        return wx.Bitmap(image)
    
    def OnButton(self, event):
        event_id = event.GetId()  # Get the ID of the event

        # Mapping of button IDs to their corresponding section objects  
        section_mapping = {
            ID_BTN_FIFTY_ONE: self.sz1,
            ID_BTN_FIFTY_TWO: self.sz2,
            ID_BTN_FIFTY_TREE: self.sz3,
            ID_BTN_FIFTY_FOUR: self.sz4,
            ID_BTN_FIFTY_FIVE: self.sz5,
        }

        # Check if the event ID corresponds to a section toggle  
        if event_id in section_mapping:
            section = section_mapping[event_id]  # Get the corresponding section  
            section.Show(not section.IsShown())   # Toggle the visibility of the section  
        elif event_id == ID_BTN_CLOSE:
            self.OnCloseWindow(self)  # Handle the close button event  
        else:
            event.Skip()  # Skip unhandled events and continue to the default handler

        self.panelSizer.Fit(self)  # Adjust the layout after any changes

    def OnLeftDown(self, event):
        self.CaptureMouse()
        x, y = self.ClientToScreen(event.GetPosition())
        originx, originy = self.GetPosition()
        dx = x - originx
        dy = y - originy
        self.delta = ((dx, dy))

    def OnLeftUp(self, event):
        if self.HasCapture():
            self.ReleaseMouse()

    def OnMouseMove(self, event):
        if event.Dragging() and event.LeftIsDown():
            x, y = self.ClientToScreen(event.GetPosition())
            fp = (x - self.delta[0], y - self.delta[1])
            self.Move(fp)

    def OnActivate(self, event):
        # Enable or disable buttons based on activation state  
        active = event.GetActive()

        if active:
            self.SetIcon(self.enableFrameIcon)
        else:
            self.SetIcon(self.disableFrameIcon)
            
        # List of buttons to enable/disable  
        buttons = [self.btn1, self.btn2, self.btn3, self.btn4, self.btn5, self.btn6]
    
        # Enable or disable each button  
        for btn in buttons:
            btn.Enable(active)
        
        # Set the foreground colour and logo bitmap based on the activation state
        self.title.SetForegroundColour("grey25" if active else "grey30")
        self.txt1.SetForegroundColour("grey25" if active else "grey30")
        self.txt2.SetForegroundColour("grey25" if active else "grey30")
        self.txt3.SetForegroundColour("grey25" if active else "grey30")
        self.txt4.SetForegroundColour("grey25" if active else "grey30")
        self.txt5.SetForegroundColour("grey25" if active else "grey30")
        self.btn6.SetForegroundColour("blue" if active else "grey30")
        self.logo.SetBitmap(self.bitmap if active else self.gray_bitmap)

        self.btn6.SetFocus()        
        self.Refresh()  # Redraw the window to reflect the changes
        
    def OnKeyUp(self, event):
        if event.GetKeyCode() == wx.WXK_ESCAPE:
            # Close the frame, no action
            self.OnCloseWindow(event)
        event.Skip()

    def OnCloseWindow(self, event):
        self.EndModal(0)
        
#-------------------------------------------------------------------------------

class BitmapPanelPdc(BufferedPanel):
    def __init__(self, parent):
        super(BitmapPanelPdc, self).__init__(parent)
        
        # Return bitmaps folder
        self.bitmaps_dir = wx.GetApp().GetBitmapsDir()

        self.bg_bmp = wx.Bitmap(os.path.join(self.bitmaps_dir,
                                             "background.jpg"),
                            type=wx.BITMAP_TYPE_JPEG)
        
        if SHOW_BACKGROUND:
            self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)
        
    def OnEraseBackground(self, event):
        dc = event.GetDC()

        # Tile the background bitmap
        try:
            sz = self.GetClientSize()
        except RuntimeError:
            return
        w = self.bg_bmp.GetWidth()
        h = self.bg_bmp.GetHeight()
        x = 0

        while x < sz.width:
            y = 0

            while y < sz.height:
                dc.DrawBitmap(self.bg_bmp, x, y)
                y = y + h

            x = x + w

#-------------------------------------------------------------------------------

class GradientPanelPdc(BufferedPanel):
    def __init__(self, parent):
        super(GradientPanelPdc, self).__init__(parent)

        # Colourdb
        wx.lib.colourdb.updateColourDB()
        
        # Simplified init method
        self.BindEvents()
        
    def BindEvents(self):
        self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)  # for GetDefaultAttributes() examples

    def OnEraseBackground(self, event):
        dc = event.GetDC()
        # '#062129', '#83F682' (Green)
        dc.GradientFillLinear(self.GetClientRect(), '#eae9e9', '#cdcdb4', wx.NORTH)

#-------------------------------------------------------------------------------

class ColourPanelPdc(BufferedPanel):
    def __init__(self, parent):
        super(ColourPanelPdc, self).__init__(parent)
        
        # Colourdb
        wx.lib.colourdb.updateColourDB()

        # Simplified init method
        self.SetProperties()
        self.BindEvents()
        
    def SetProperties(self):
        self.SetBackgroundColour("LIGHT BLUE")  # for GetDefaultAttributes() examples
        
    def BindEvents(self):
        self.Bind(wx.EVT_PAINT, self.OnPaint)

    def OnPaint(self, event):
        dc = wx.PaintDC(self)
        # Clear the buffer
        dc.SetBackground(wx.Brush("LIGHT BLUE"))  #535353 (Dark grey)
        dc.Clear()

#-------------------------------------------------------------------------------

class ColourPanel(BufferedPanel):
    def __init__(self, parent):
        super(ColourPanel, self).__init__(parent)
        
        # Colourdb
        wx.lib.colourdb.updateColourDB()
        
        # Simplified init method
        self.SetProperties()
        
    def SetProperties(self):
        # ROSYBROWN | YELLOWGREEN | GREY90 | #282828
        self.SetBackgroundColour("#202020")   

#-------------------------------------------------------------------------------
        
class DefaultPanel(wx.Panel):
    def __init__(self, parent):
        super(DefaultPanel, self).__init__(parent, style=wx.TAB_TRAVERSAL |
                                           wx.NO_FULL_REPAINT_ON_RESIZE)

        # Return bitmaps folder
        self.bitmaps_dir = wx.GetApp().GetBitmapsDir()
        
        # Load bitmap
        self.wx_bitmap = wx.Bitmap(os.path.join(self.bitmaps_dir,
                                              "wxpython.png"),
                                 type=wx.BITMAP_TYPE_PNG)
        self.width_a = self.wx_bitmap.GetWidth()
        self.height_a = self.wx_bitmap.GetHeight()

        self.os_bitmap = wx.Bitmap(os.path.join(self.bitmaps_dir,
                                                "os.png"),
                                 type=wx.BITMAP_TYPE_PNG)
        self.width_b = self.os_bitmap.GetWidth()
        self.height_b = self.os_bitmap.GetHeight()
        
        self.Bind(wx.EVT_PAINT, self.OnPaint)

    def OnPaint(self, event):
        w, h = self.GetClientSize()
        
        dc = wx.BufferedPaintDC(self)
        
        # Clear the buffer
        dc.Clear()
        gcdc = wx.GCDC(dc)

        gc = wx.GraphicsContext.Create(dc) 
        
        gcdc.GradientFillLinear(self.GetClientRect(), '#919191', '#dddddd', wx.NORTH) 

        # Define the points of the triangle
        base_length = 100
        height_triangle = 50
        
        # Calculate points based on window size
        point1 = (w // 2 - base_length // 2-170, h // 2 + height_triangle // 2-100)  # Left vertex
        point2 = (w // 2 + base_length // 2, h // 2 + height_triangle // 2+250)  # Right vertex
        point3 = (w // 2+180, h // 2 - height_triangle-120)  # Vertex

        # Draw the triangle
        gcdc.SetBrush(wx.Brush(wx.Colour(120, 120, 120, 120)))       
        gcdc.SetPen(wx.Pen(wx.Colour(0, 0, 0, 20), 20)) 
        gcdc.DrawPolygon([point1, point2, point3])

        # Convert bitmap to grayscale 
        gray_bitmap = self.ConvertToGrayscale(self.wx_bitmap)

        # Draw the wx bitmap in grayscale  
        gcdc.DrawBitmap(gray_bitmap, int(w//2-self.width_a//2), int(h//2-self.height_a//2), True)
        gcdc.SetTextForeground(wx.SystemSettings.GetColour(wx.SYS_COLOUR_GRAYTEXT))            

        # Draw the os bitmap in grayscale  
        gcdc.DrawBitmap(self.os_bitmap, int(11), int(h-self.height_b//2-80), True)

        if not self.IsEnabled():
            if  wx.Platform == "__WXMAC__":
                # gcdc.SetTextForeground(wx.SystemSettings.GetColour(wx.SYS_COLOUR_GRAYTEXT
                gcdc.SetTextForeground("#707070")  
            else:
                gcdc.SetTextForeground(wx.SystemSettings.GetColour(wx.SYS_COLOUR_GRAYTEXT))                
        else:
            gcdc.SetTextForeground("#000000")
            
        text = "Make different backgrounds to test\ntransparency and appearance of controls."
        gcdc.DrawText(text, 20, 20)

    def ConvertToGrayscale(self, bitmap):
        # Convert bitmap to image  
        image = bitmap.ConvertToImage()
    
        # Set a lightening factor  
        brighten_factor = 10  # You can adjust this value according to your needs
    
        # Convert image to grayscale 
        for x in range(image.GetWidth()):
            for y in range(image.GetHeight()):
                r = image.GetRed(x, y)
                g = image.GetGreen(x, y)
                b = image.GetBlue(x, y)
                gray = int(0.299 * r + 0.587 * g + 0.114 * b)  # Brightness method
            
                # Add the lightening factor while making sure not to exceed 255  
                gray = min(255, gray + brighten_factor)
            
                image.SetRGB(x, y, gray, gray, gray)

        # Create a bitmap from the edited image  
        return wx.Bitmap(image)
        
#-------------------------------------------------------------------------------

class MainPanel(ScrolledPanel):
    def __init__(self, parent): 
        super(MainPanel, self).__init__(parent)

        # Initialisation  
        self.focused_ctrl = False
        
        # Return application name
        self.app_name = wx.GetApp().GetAppName()
        # Return bitmaps folder
        self.bitmaps_dir = wx.GetApp().GetBitmapsDir()
        # Return icons folder
        self.icons_dir = wx.GetApp().GetIconsDir()
        
        self.sizer = wx.BoxSizer()
        self.panel = DefaultPanel(self)
        if wx.Platform in ['__WXMAC__']:
            self.panel.SetToolTip(MACOS_MESSAGE)
            
        self.sizer.Add(self.panel, 1, wx.EXPAND)
        self.SetSizer(self.sizer)
        
    def SetPanel(self, ptype):
        if ptype == "BITMAP_PDC":
            self.newpanel = BitmapPanelPdc(self)
            self.newpanel.Refresh()
        elif ptype == "GRADIENT_PDC":
            self.newpanel = GradientPanelPdc(self)
            self.newpanel.Refresh()
        elif ptype == "COLOUR_PDC":
            self.newpanel = ColourPanelPdc(self)
            self.newpanel.Refresh()
        elif ptype == "SOLID_COLOUR":
            self.newpanel = ColourPanel(self)
            self.newpanel.Refresh()

        self.CreateCtrls()
        self.BindEvents()
        self.DoLayout()

    def CreateCtrls(self):
        # System font size
        ft_size = self.GetButtonFontSize()
            
        self.button1 = Button(self.newpanel, id=ID_BTN_ONE, label="Button 1", normalLabelColour="#ffffff",
                              hoverLabelColour="#ffffff", pressedLabelColour="#ffffff",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                              normalBtnColour=(wx.Colour(146, 193, 48, 255), wx.Colour(75, 120, 21, 255)), 
                              hoverBtnColour="#92c130",
                              pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                              borderColour="#ffffff",
                              focusColour="#fdf240",
                              indicatorColour="#ffffff", appearance=BTN_RND_RECTANGLE,  
                              borderSize=3, size=(100, 30),
                              )
        #self.button1.SetToolTip("This is button 1")
        #self.button1.Enable(False)
        self.button1.SetFocus()
        
        self.button2 = Button(self.newpanel, id=ID_BTN_TWO, label="Button 2", normalLabelColour="#46a0f5",
                              hoverLabelColour="#ffffff", pressedLabelColour="#ffffff",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_NORMAL,
                              normalBtnColour=(wx.Colour(255, 255, 255, 255), wx.Colour(255, 255, 255, 255)), 
                              hoverBtnColour="#46a0f5",
                              pressedBtnColour="#1473e6", disabledBtnColour="#838383",
                              borderColour="#46a0f5",
                              focusColour="#1473e6",
                              indicatorColour="#ffffff", appearance=BTN_RND_RECTANGLE, 
                              borderSize=3, size=(100, 30),
                              image="./bitmaps/item_about.png", 
                              image_position='left' 
                              )
        #self.button2.SetToolTip("This is button 2")
        #self.button2.Enable(False)
        
        self.button3 = Button(self.newpanel, id=ID_BTN_THREE, label="Button 3", normalLabelColour="#ffffff",
                              hoverLabelColour="#000000", pressedLabelColour="#ffffff",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_NORMAL,
                              normalBtnColour=(wx.Colour(146, 193, 48, 255), wx.Colour(75, 120, 21, 255)), 
                              hoverBtnColour=(wx.Colour(146, 193, 48, 80)), 
                              pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                              borderColour="#ffffff",
                              focusColour="#fdf240",
                              indicatorColour="#ffffff", appearance=BTN_RND_RECTANGLE,  
                              borderSize=1, size=(100, 30),
                              image="./bitmaps/item_empty.png", 
                              image_position='right' 
                              )
        #self.button3.SetToolTip("This is button 3")
        #self.button3.Enable(False)

        self.button4 = Button(self.newpanel, id=ID_BTN_FOUR, label="Button 4", normalLabelColour="#ffffff",
                              hoverLabelColour="#000000", pressedLabelColour="#ffffff",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                              normalBtnColour=(wx.Colour(146, 193, 48, 255), wx.Colour(75, 120, 21, 255)), 
                              hoverBtnColour=(wx.Colour(146, 193, 48, 80)), 
                              pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                              borderColour="#ffffff",
                              focusColour="#fdf240",
                              indicatorColour="#ffffff", appearance=BTN_RECTANGLE,  
                              borderSize=3, size=(100, 30),
                              image="./bitmaps/item_empty.png", 
                              image_position='left' 
                              )
        #self.button4.SetToolTip("This is button 4")
        #self.button4.Enable(False)
        
        self.button5 = Button(self.newpanel, id=ID_BTN_FIVE, label="Button 5", normalLabelColour="#000000",
                              hoverLabelColour="#000000", pressedLabelColour="#000000",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                              normalBtnColour=(wx.Colour(146, 193, 48, 255), wx.Colour(75, 120, 21, 255)), 
                              hoverBtnColour="#92c130",
                              pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                              borderColour="#ffffff",
                              focusColour="#fdf240",
                              indicatorColour="#ffffff", appearance=BTN_RECTANGLE,
                              borderSize=3, size=(100, 30),
                              )
        #self.button5.SetToolTip("This is button 5")
        #self.button5.Enable(False)
        
        self.button6 = Button(self.newpanel, id=ID_BTN_SIX, label="Button 6", normalLabelColour="#46a0f5",
                              hoverLabelColour="#ffffff", pressedLabelColour="#ffffff",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                              normalBtnColour=(wx.Colour(255, 255, 255, 255), wx.Colour(255, 255, 255, 255)),
                              hoverBtnColour="#46a0f5",
                              pressedBtnColour="#1473e6", disabledBtnColour="#838383",
                              borderColour="#46a0f5",
                              focusColour="#1473e6",
                              indicatorColour="#ffffff", appearance=BTN_RECTANGLE,
                              borderSize=3, size=(120, 50),
                              image="./bitmaps/phoenix_title.png", 
                              image_position='left' 
                              )
        #self.button6.SetToolTip("This is button 6")
        #self.button6.Enable(False)
        
        self.button7 = Button(self.newpanel, id=ID_BTN_SEVEN, label="Button 7", normalLabelColour="#ffffff",
                              hoverLabelColour="#000000", pressedLabelColour="#ffffff",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                              normalBtnColour=(wx.Colour(146, 193, 48, 255), wx.Colour(75, 120, 21, 255)), 
                              hoverBtnColour="#92c130",
                              pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                              borderColour="#ffffff",
                              focusColour="#fdf240",
                              indicatorColour="#ffffff", appearance=BTN_RECTANGLE,
                              borderSize=3, size=(100, 30),
                              image="./bitmaps/item_empty.png", 
                              image_position='left' 
                              )
        #self.button7.SetToolTip("This is button 7")
        #self.button7.Enable(False)

        self.button8 = Button(self.newpanel, id=ID_BTN_EIGHT, label="Button 8", normalLabelColour="#ffffff",
                              hoverLabelColour="#000000", pressedLabelColour="#ffffff",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                              normalBtnColour=(wx.Colour(146, 193, 48, 255), wx.Colour(75, 120, 21, 255)), 
                              hoverBtnColour="#92c130",
                              pressedBtnColour="#4b7815", disabledBtnColour="#838383",
                              borderColour="#ffffff",
                              focusColour="#fdf240",
                              indicatorColour="#ffffff", appearance=BTN_RECTANGLE,
                              borderSize=3, size=(120, 55),
                              image="./bitmaps/item_quit.png", 
                              image_position='top' 
                              )
        #self.button8.SetToolTip("This is button 8")
        #self.button8.Enable(False)
        
        self.button9 = Button(self.newpanel, id=ID_BTN_NINE, label="Button 9", normalLabelColour="#000000",
                              hoverLabelColour="#000000", pressedLabelColour="#000000",
                              font_size=ft_size, font_weight=wx.FONTWEIGHT_NORMAL,
                              normalBtnColour=(wx.Colour(254, 198, 3, 255), wx.Colour(221, 140, 14, 255)), 
                              hoverBtnColour="#fec603", 
                              pressedBtnColour="#f45d1a", disabledBtnColour="#838383",
                              borderColour="#ffffff",
                              focusColour="#ffffff",
                              indicatorColour="#000000", appearance=BTN_RND_RECTANGLE,  
                              borderSize=1, size=(100, 30),
                              )
        #self.button9.SetToolTip("This is button 9")
        #self.button9.Enable(False)
        
        self.button10 = Button(self.newpanel, id=ID_BTN_TEN, label="Button 10\nwith image\nand\nmultiline",
                               normalLabelColour="#ffffff", hoverLabelColour="#ffffff", pressedLabelColour="#6450cd",
                               font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                               normalBtnColour=(wx.Colour(135, 112, 254, 255), wx.Colour(135, 112, 254, 255)),
                               hoverBtnColour="#6450cd",
                               pressedBtnColour="#ffffff", disabledBtnColour="#838383",
                               borderColour="#6450cd",
                               focusColour="#ccc2ff",
                               appearance=BTN_RND_RECTANGLE,
                               borderSize=1, size=(180, 100),
                               image="./bitmaps/filesave.png", 
                               image_position='left', 
                               text_alignment='left',
                               )
        #self.button10.SetToolTip("This is button 10")
        #self.button10.Enable(False)

        self.button11 = Button(self.newpanel, id=ID_BTN_ELEVEN, label="Button 11\nHello", normalLabelColour="#ffffff",
                               hoverLabelColour="#ffffff", pressedLabelColour="#ffffff",
                               font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                               normalBtnColour=(wx.Colour(90, 173, 255, 255), wx.Colour(5, 114, 255, 255)),
                               hoverBtnColour="#46a0f5",
                               pressedBtnColour="#1473e6", disabledBtnColour="#838383",
                               borderColour="#ffffff",
                               focusColour="#89c0f7",
                               appearance=BTN_RND_RECTANGLE,
                               borderSize=3, size=(100, 100),
                               image="./bitmaps/filesave.png", 
                               image_position='bottom', 
                               text_alignment='center',
                               )
        #self.button11.SetToolTip("This is button 11")
        #self.button11.Enable(False)

        self.button12 = Button(self.newpanel, id=ID_BTN_TWELVE, label="Button 12", normalLabelColour="yellow",
                               hoverLabelColour="#ffffff", pressedLabelColour="gray",
                               font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                               normalBtnColour=(wx.Colour(90, 173, 255, 255), wx.Colour(5, 114, 255, 255)),
                               hoverBtnColour="#46a0f5",
                               pressedBtnColour="#1473e6", disabledBtnColour="#838383",
                               borderColour="#ffffff",
                               focusColour="#89c0f7",
                               appearance=BTN_RND_RECTANGLE,
                               borderSize=3, size=(120, 50),
                               image="./bitmaps/filesave.png", 
                               image_position='right', 
                               )
        #self.button12.SetToolTip("This is button 12")
        #self.button12.Enable(False)
        
        self.button13 = Button(self.newpanel, id=ID_BTN_THIRTEEN, label="Button 13", normalLabelColour="#ec008c",
                               hoverLabelColour="#00aeef", pressedLabelColour="#faa61a",
                               font_size=ft_size, font_weight=wx.FONTWEIGHT_NORMAL,
                               normalBtnColour=(wx.Colour(0, 0, 0, 0), wx.Colour(0, 0, 0, 0)),
                               hoverBtnColour=wx.Colour(0, 0, 0, 0),
                               pressedBtnColour=wx.Colour(0, 0, 0, 0), disabledBtnColour=wx.Colour(0, 0, 0, 0),
                               borderColour=wx.Colour(0, 0, 0, 0),
                               focusColour=wx.Colour(255, 255, 255, 0),
                               indicatorColour="magenta", appearance=BTN_RND_RECTANGLE,
                               borderSize=1, size=(100, 30),
                               )
        #self.button13.SetToolTip("This is button 13")
        #self.button13.Enable(False)

        self.button14 = Button(self.newpanel, id=ID_BTN_FOURTEEN, label="Button 14", normalLabelColour="#838383",
                               hoverLabelColour="magenta", pressedLabelColour="#6450cd",
                               font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                               normalBtnColour=(wx.Colour(0, 0, 0, 0), wx.Colour(0, 0, 0, 0)),
                               hoverBtnColour=wx.Colour(0, 0, 0, 0),
                               pressedBtnColour=wx.Colour(0, 0, 0, 0), disabledBtnColour=wx.Colour(0, 0, 0, 0),
                               borderColour=wx.Colour(0, 0, 0, 0),
                               focusColour=wx.Colour(150, 150, 150, 255),
                               indicatorColour=wx.Colour(0, 0, 0, 255), appearance=BTN_RND_RECTANGLE,
                               borderSize=0, size=(110, 30),
                               image="./bitmaps/folder.png", 
                               image_position='left' 
                               )
        #self.button14.SetToolTip("This is button 14")
        #self.button14.Enable(False)

        self.button15 = Button(self.newpanel, id=ID_BTN_FIFTEEN, label="Button 15", normalLabelColour="#838383",
                               hoverLabelColour="magenta", pressedLabelColour="#6450cd",
                               font_size=ft_size, font_weight=wx.FONTWEIGHT_BOLD,
                               normalBtnColour=(wx.Colour(0, 0, 0, 0), wx.Colour(0, 0, 0, 0)),
                               hoverBtnColour=wx.Colour(0, 0, 0, 0),
                               pressedBtnColour=wx.Colour(0, 0, 0, 0), disabledBtnColour=wx.Colour(0, 0, 0, 0),
                               borderColour="#8770fe",
                               focusColour="#8770fe",
                               indicatorColour=wx.Colour(0, 0, 0, 0), appearance=BTN_RND_RECTANGLE,
                               borderSize=1, size=(110, 30),
                               image="./bitmaps/folder.png", 
                               image_position='left' 
                               )
        #self.button15.SetToolTip("This is button 15")
        #self.button15.Enable(False)
        
    def BindEvents(self):
        # List of buttons to bind the button click event  
        buttons = [self.button1,  self.button2,  self.button3, 
                   self.button4,  self.button5,  self.button6, 
                   self.button7,  self.button8,  self.button9,
                   self.button10, self.button11, self.button12,
                   self.button13, self.button14, self.button15,
        ]

        # Bind the EVT_BUTTON event for each button in the list 
        for button in buttons:
            button.Bind(wx.EVT_BUTTON, self.OnButton)  

        self.newpanel.Bind(wx.EVT_LEFT_DOWN, self.OnPanelClick)  # Event to handle focus when clicking on the title bar 
        self.Bind(wx.EVT_BUTTON, self.OnControlClick)  # Event to manage focus when clicking on the panel
        self.Bind(wx.EVT_ACTIVATE, self.OnActivate)
        
    def DoLayout(self):
        mainSizer = wx.BoxSizer(wx.VERTICAL)

        self.btnSizerA = wx.BoxSizer(wx.HORIZONTAL)
        self.btnSizerA.Add(self.button1, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerA.Add(self.button2, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerA.Add(self.button3, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerA.Add(self.button4, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        
        self.btnSizerB = wx.BoxSizer(wx.HORIZONTAL)
        self.btnSizerB.Add(self.button5, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerB.Add(self.button6, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerB.Add(self.button7, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerB.Add(self.button8, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        
        self.btnSizerC = wx.BoxSizer(wx.HORIZONTAL)
        self.btnSizerC.Add(self.button9, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerC.Add(self.button10, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerC.Add(self.button11, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerC.Add(self.button12, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        
        self.btnSizerD = wx.BoxSizer(wx.HORIZONTAL)
        self.btnSizerD.Add(self.button13, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerD.Add(self.button14, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        self.btnSizerD.Add(self.button15, 0, wx.ALIGN_TOP | wx.ALIGN_CENTER | wx.ALL, 2)
        
        mainSizer.AddStretchSpacer()
        mainSizer.Add(self.btnSizerA, 0, wx.ALIGN_CENTER) 
        mainSizer.AddStretchSpacer()
        mainSizer.Add(self.btnSizerB, 0, wx.ALIGN_CENTER) 
        mainSizer.AddStretchSpacer()
        mainSizer.Add(self.btnSizerC, 0, wx.ALIGN_CENTER) 
        mainSizer.AddStretchSpacer()
        mainSizer.Add(self.btnSizerD, 0, wx.ALIGN_CENTER) 
        mainSizer.AddStretchSpacer()
        self.newpanel.SetSizerAndFit(mainSizer) 

        self.sizer.Replace(self.panel, self.newpanel) 
        self.panel.Destroy() 
        self.panel = self.newpanel    
        self.Layout()   

    def GetButtonFontSize(self):
        font = wx.SystemSettings.GetFont(wx.SYS_DEFAULT_GUI_FONT)
        font_size = font.GetPointSize()
    
        # Dictionary for size adjustments by platform  
        platform_adjustments = {
            '__WXMSW__': 1,
            '__WXGTK__': -1,
            '__WXMAC__': 2  
        }
    
        # Use get() to get the fit, or 0 if the platform is not specified
        adjustment = platform_adjustments.get(wx.Platform, 0)
  
        return font_size + adjustment  # Returns the adjusted font size
        
    def OnButton(self, event):
        event_id = event.GetId()  # Get the ID of the event

        # Mapping of button IDs to their respective label strings  
        button_labels = {
            ID_BTN_ONE: "ID_BTN_ONE",
            ID_BTN_TWO: "ID_BTN_TWO",
            ID_BTN_THREE: "ID_BTN_THREE",
            ID_BTN_FOUR: "ID_BTN_FOUR",
            ID_BTN_FIVE: "ID_BTN_FIVE",
            ID_BTN_SIX: "ID_BTN_SIX",
            ID_BTN_SEVEN: "ID_BTN_SEVEN",
            ID_BTN_EIGHT: "ID_BTN_EIGHT",
            ID_BTN_NINE: "ID_BTN_NINE",
            ID_BTN_TEN: "ID_BTN_TEN",
            ID_BTN_ELEVEN: "ID_BTN_ELEVEN",
            ID_BTN_TWELVE: "ID_BTN_TWELVE",
            ID_BTN_THIRTEEN: "ID_BTN_THIRTEEN",
            ID_BTN_FOURTEEN: "ID_BTN_FOURTEEN",
            ID_BTN_FIFTEEN: "ID_BTN_FIFTEEN",
        }

        # Check if the event ID is in the button labels mapping  
        if event_id in button_labels:
            # Print the button ID and its label  
            print(f"{button_labels[event_id]} :", event_id)  
            print("Label :", event.GetEventObject().GetLabel(), "\n")
        event.Skip()  # Skip any unhandled events and continue to the default handler

    def OnControlClick(self, event):  
        # Manage focus on controls  
        self.focused_ctrl = event.GetEventObject()
        self.focused_ctrl.SetFocus()

    def OnPanelClick(self, event): 
        # Set focus on the control when the panel is clicked   
        self.button1.SetFocus()
        # Set focus on the control when the window is activated
        if self.focused_ctrl:  # Give focus to the last selected control  
            self.focused_ctrl.SetFocus()
        event.Skip()  # Allows events to continue to be processed

    def OnActivate(self, event): 
        if event.GetActive():
            self.button1.SetFocus()
            # Set focus on the control when the window is activated  
            if self.focused_ctrl:  # Give focus to the last selected control  
                self.focused_ctrl.SetFocus()
            self.Refresh()
        else:
            pass
        event.Skip()
        
#-------------------------------------------------------------------------------

class Frame(wx.Frame):
    def __init__(self, id, title):
        super().__init__(None, -1, title, size=(-1, -1),
                         style=wx.DEFAULT_FRAME_STYLE |
                         wx.NO_FULL_REPAINT_ON_RESIZE)

        # Attribute to check if the Active button exists
        self.btnToggle = None
        
        # Return application name
        self.app_name = wx.GetApp().GetAppName()
        # Return bitmaps folder
        self.bitmaps_dir = wx.GetApp().GetBitmapsDir()
        # Return icons folder
        self.icons_dir = wx.GetApp().GetIconsDir()
        
        # Simplified init method        
        self.SetProperties()
        self.MakeMenuBar()
        self.MakeStatusBar()
        self.CreateCtrls()
        self.BindEvents()
        self.DoLayout()

    def SetProperties(self):
        self.enableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                    "enabled_icon.ico"),
                                       type=wx.BITMAP_TYPE_ICO)
        self.SetIcon(self.enableFrameIcon)

        self.disableFrameIcon = wx.Icon(os.path.join(self.icons_dir,
                                                     "disabled_icon.ico"),
                                        type=wx.BITMAP_TYPE_ICO)

    def MakeMenuBar(self):
        # Define the bitmap file names and corresponding IDs and menu texts  
        bitmap_items = {
            "item_quit.png": (ID_EXIT, "&Quit\tCtrl+Q", "Quit the application."),
            "item_system.png": (ID_SYSTEM, "S&ystem\tCtrl+Y", "System information about this demo."),
            "item_thanks.png": (ID_THANKS_TO, "Thanks to...\tCtrl+T", "Show Thanks to dialog."),
            "item_about.png": (ID_ABOUT, "A&bout\tCtrl+W", "About this program."),
            "item_bitmap.png": (ID_BITMAP_PDC, "Bitmap pdc\tCtrl+B", "Show pdc bitmap background."),
            "item_gradient.png": (ID_GRADIENT_PDC, "Gradient pdc\tCtrl+G", "Show pdc gradient background."),
            "item_colour.png": (ID_COLOUR_PDC, "Colour pdc\tCtrl+C", "Show pdc colour background."),
            "item_solid_colour.png": (ID_SOLID_COLOUR, "Solid colour\tCtrl+D", "Show solid colour background."),
        }

        # Create menus  
        file_menu = wx.Menu()
        test_menu = wx.Menu()
        system_menu = wx.Menu()
        about_menu = wx.Menu()

        # Create menu items dynamically based on the bitmap_items dictionary  
        for bitmap_name, (item_id, label, help_string) in bitmap_items.items():
            item_bitmap = wx.Bitmap(os.path.join(self.bitmaps_dir, bitmap_name), type=wx.BITMAP_TYPE_PNG)
            menu_item = wx.MenuItem(
                file_menu if item_id == ID_EXIT else 
                test_menu if item_id in [ID_BITMAP_PDC, ID_GRADIENT_PDC, ID_COLOUR_PDC, ID_SOLID_COLOUR] else 
                system_menu if item_id == ID_SYSTEM else 
                about_menu, 
                item_id, label, help_string  
            )
            menu_item.SetBitmap(item_bitmap)

            # Append to the appropriate menu  
            if item_id == ID_EXIT:
                file_menu.Append(menu_item)
            elif item_id in [ID_BITMAP_PDC, ID_GRADIENT_PDC, ID_COLOUR_PDC, ID_SOLID_COLOUR]:
                test_menu.Append(menu_item)
            elif item_id == ID_SYSTEM:
                system_menu.Append(menu_item)
            elif item_id in [ID_THANKS_TO, ID_ABOUT]:
                about_menu.Append(menu_item)

        # Create the menu bar and append menus  
        menu_bar = wx.MenuBar()
        menu_bar.Append(file_menu, '&File')
        menu_bar.Append(test_menu, '&Test')
        menu_bar.Append(system_menu, '&System')
        menu_bar.Append(about_menu, '&Help')

        # Set the menu bar  
        self.SetMenuBar(menu_bar)

    def MakeStatusBar(self):
        self.CreateStatusBar(2)
    
        # Set the widths of status bar fields  
        self.SetStatusWidths([-4, -1])
    
        # Set welcome message in the first field  
        self.SetStatusText('Welcome!', 0)
    
        # Set locale to the user's default (for date formatting)
        locale.setlocale(locale.LC_TIME, '')  # Use '' to set to the user's default locale

        # Get today's date and format it according to the locale  
        today = datetime.datetime.today()
        formatted_date = today.strftime('%A, %d %B %Y')  # Example format: "Monday, 01 January 2023"

        # Capitalize the first letter of the formatted date  
        formatted_date = formatted_date.capitalize()  # Capitalizing the first letter
    
        # Set formatted date in the second field of the status bar  
        self.SetStatusText(formatted_date, 1)
        
    def CreateCtrls(self):
        self.mainPanel = BufferedPanel(self)       
        self.panel = MainPanel(self.mainPanel)
        
    def BindEvents(self):        
        self.Bind(wx.EVT_ACTIVATE, self.OnActivate)
        self.Bind(wx.EVT_MENU, self.OnMenu) 
        self.Bind(wx.EVT_WINDOW_MODAL_DIALOG_CLOSED, self.OnWindowModalDialogClosed) 
        self.Bind(wx.EVT_CHAR_HOOK, self.OnKeyUp)
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)

    def DoLayout(self):
        self.sizerBtn = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        
        mainSizer.Add(self.panel, 1, wx.EXPAND)
        mainSizer.Add(self.sizerBtn, 0, wx.ALL|wx.EXPAND, 5)

        self.mainPanel.SetSizer(mainSizer)

    def OnChangePanel(self, event):
        if not self.btnToggle:
            self.btnClose = wx.Button(self.mainPanel, label="&Close")
            self.Bind(wx.EVT_BUTTON, self.OnBtnClose, self.btnClose)
            self.btnToggle = wx.Button(self.mainPanel, label="&Disable")
            self.Bind(wx.EVT_BUTTON, self.OnCtrlEnable, self.btnToggle)
            self.sizerBtn.Add(self.btnClose, 1, wx.ALL | wx.CENTER, 5)
            self.sizerBtn.Add(self.btnToggle, 1, wx.ALL | wx.CENTER, 5)
            self.mainPanel.Layout()
            self.btnClose.Enable(True)

    def OnbuttonClick(self, event):
        # Manage focus on buttons  
        self.focused_button = event.GetEventObject()
        self.focused_button.SetFocus()

    def OnActivate(self, event):
        if event.GetActive():
            self.SetIcon(self.enableFrameIcon)
            if self.btnToggle:
                self.btnToggle.Enable(True)
            self.Refresh()
        else:
            self.SetIcon(self.disableFrameIcon)
            if self.btnToggle:
                self.btnToggle.Enable(False)
            self.Refresh()
        event.Skip()
        
    def OnMenu(self, event):        
        event_id = event.GetId()

        # Handle specific menu actions  
        if event_id == ID_EXIT:
            self.OnCloseWindow(event)  # Close the application  
        elif event_id == ID_SYSTEM:
            self.OnSystemDialog(event)      # Open the system window  
        elif event_id == ID_THANKS_TO:
            self.OnThanksToDialog(event)      # Show the "Thanks To" information  
        elif event_id == ID_ABOUT:
            self.OnAboutDialog(event)         # Show the about dialog  
        elif event_id in [ID_GRADIENT_PDC, ID_BITMAP_PDC, ID_COLOUR_PDC, ID_SOLID_COLOUR]:
            # Common actions for panel changes  
            self.OnChangePanel(self)    # Change the current panel  
            self.btnToggle.Enable()      # Enable the toggle button

            # Mapping event IDs to panel names  
            panel_mapping = {
                ID_GRADIENT_PDC: "GRADIENT_PDC",
                ID_BITMAP_PDC: "BITMAP_PDC",
                ID_COLOUR_PDC: "COLOUR_PDC",
                ID_SOLID_COLOUR: "SOLID_COLOUR",
            }

            # Update panel with matching ID  
            self.panel.SetPanel(panel_mapping[event_id])
            # Force the scrollbar to be displayed (hack) 
            self.panel.SendSizeEvent()
        else:
            event.Skip()  # Skip unhandled events

    def OnCtrlEnable(self, event):
        # Determine the current state based on the toggle label  
        is_disabled = self.btnToggle.GetLabel() == "&Disable"
    
        # List of buttons to enable/disable  
        buttons = [
            self.panel.button1,   self.panel.button2,  self.panel.button3,
            self.panel.button4,   self.panel.button5,  self.panel.button6,
            self.panel.button7,   self.panel.button8,  self.panel.button9,
            self.panel.button10,  self.panel.button11, self.panel.button12,
            self.panel.button13,  self.panel.button14, self.panel.button15,
        ]

        # Set the enable state for each button  
        for btn in buttons:
            btn.Enable(not is_disabled)

        # Update the toggle button label  
        self.btnToggle.SetLabel("&Enable" if is_disabled else "&Disable")
            
    def OnSystemDialog(self, event):
        self.dialogSystem = SystemDialog(self,
                                         id=ID_SYSTEM, 
                                         title="System...",
                                         style=wx.DEFAULT_DIALOG_STYLE)
        self.dialogSystem.CenterOnParent()

        # This does not return until the dialog is closed
        val = self.dialogSystem.ShowModal()

        if val == wx.ID_OK:
            print("You pressed OK\n")
        else:
            print("You pressed Cancel\n")

        self.dialogSystem.Destroy()
        
    def OnThanksToDialog(self, event):
        # Correctly define the path to the image  
        image_path = os.path.join(self.bitmaps_dir, "wxpython.png")  
        # or without image
        # image_path = None
        self.dialogThanksTo = ThanksToDialog(self,
                                             id=-1,
                                             message=THANKS_MESSAGE,
                                             backgroundColour=(wx.Colour(0, 0, 0, 255)),
                                             fading=0,  
                                             image_path=image_path,   
                                             title="Thanks to...",
                                             style=wx.DEFAULT_DIALOG_STYLE)
        if wx.Platform in ['__WXMAC__']:
            self.dialogThanksTo.SetScrollSpeed(0.3)
        else:
            self.dialogThanksTo.SetScrollSpeed(0.5)
            
        self.dialogThanksTo.CenterOnParent()

        # This does not return until the dialog is closed
        val = self.dialogThanksTo.ShowModal()

        if val == wx.ID_OK:
            print("You pressed OK\n")
        else:
            print("You pressed Cancel\n")

        self.dialogThanksTo.Destroy()
        
    def OnAboutDialog(self, event):
        self.dialogAbout = AboutDialog(self,
                                       id=-1,
                                       title="About...",
                                       style=wx.DEFAULT_DIALOG_STYLE)
        self.dialogAbout.CenterOnParent()

        # This does not return until the dialog is closed
        val = self.dialogAbout.ShowModal()

        if val == wx.ID_OK:
            print("You pressed OK\n")
        else:
            print("You pressed Cancel\n")

        self.dialogAbout.Destroy()

    def OnWindowModalDialogClosed(self, event): 
        dialog = event.GetDialog()
        val = event.GetReturnCode()
        dialog.Destroy()
        
    def OnKeyUp(self, event):
        if event.GetKeyCode() == wx.WXK_ESCAPE:
            # Close the frame, no action
            self.OnCloseWindow(event)
        event.Skip()
        
    def OnBtnClose(self, event):
        print("Id : (%d)\n" % self.GetId())
        self.Close()

    def OnCloseWindow(self, event):
        self.Destroy()

#-------------------------------------------------------------------------------

class App(wx.App, wx.lib.mixins.inspection.InspectionMixin):
    def OnInit(self):
        
        self.installDir = os.path.split(os.path.abspath(sys.argv[0]))[0]

        # Gather system and library information  
        osname = platform.system()
        python = sys.version.split()[0]
        wxpython = wx.VERSION_STRING[:5]
        wxwidgets = wx.GetLibraryVersionInfo().VersionString[10:]  
        encoding = sys.getfilesystemencoding()
        name = "Button demo"
        version = VERSION

        # Information to be printed  
        info = [
            ("Platform", osname),
            ("Python", python),
            ("wxPython", wxpython),
            ("wxWidgets", wxwidgets),
            ("Encoding", encoding + "\n"),
            ("Name", name),
            ("Version", version)
        ]

        # Print the gathered information 
        print("")
        for label, value in info:
            print(f"{label} : {value}")
        print("")

        self.app_name = ("%s" % (name))
        self.SetAppName(self.app_name)

        # Simplified init method for
        # the InspectionMixin base class
        self.InitInspection()

        frame = Frame(-1, self.app_name + " v" + version)
        self.SetTopWindow(frame)
        frame.SetSize((600, 600))
        frame.Center()
        frame.Show(True)

        return True

    def GetInstallDir(self):
        """
        Returns the installation directory for my application.
        """

        return self.installDir    

    def GetIconsDir(self):
        """
        Returns the icons directory for my application.
        """

        icons_dir = os.path.join(self.installDir, "icons")
        return icons_dir

    def GetBitmapsDir(self):
        """
        Returns the bitmaps directory for my application.
        """

        bitmaps_dir = os.path.join(self.installDir, "bitmaps")
        return bitmaps_dir

    def GetHtmlDir(self):
        """
        Returns the html directory for my application.
        """
        
        html_dir = os.path.join(self.installDir, "html")   
        return html_dir
    
    def GetVersion(self):
        """
        Return the current version for my application.
        """

        return __version__

#-------------------------------------------------------------------------------

def main():
    app = App(False)
    app.MainLoop()

#-------------------------------------------------------------------------------

if __name__ == "__main__" :
    main()

